/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2023
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */

package net.bluemind.cli.auditlog;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import net.bluemind.cli.cmd.api.ICmdLet;
import net.bluemind.cli.cmd.api.ICmdLetRegistration;
import net.bluemind.core.auditlogs.AuditLogEntry;
import net.bluemind.core.auditlogs.AuditLogQuery;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.domain.api.Domain;
import picocli.CommandLine.Command;
import picocli.CommandLine.Option;

@Command(name = "calendar", description = "get calendar data")
public class GetCalendarAuditLogCommand extends AbstractGetAuditLogCommand {

	private static final String[] HEADERS = { "Timestamp", "Action", "Action responsible", "Action origin",
			"Calendar name", "Calendar owner", "Calendar uid", "Event name", "Event key", "Event author",
			"Event recipients", "Event start date", "Event end date", "Update message" };
	private static final String LOG_TYPE = "calendar";

	@Option(names = "--eventName", required = false, description = "the event name to query")
	public String eventName;

	@Option(names = "--eventKey", required = false, description = "the event key to query")
	public String eventKey;

	public static class Reg implements ICmdLetRegistration {

		@Override
		public Optional<String> group() {
			return Optional.of("auditlog");
		}

		@Override
		public Class<? extends ICmdLet> commandClass() {
			return GetCalendarAuditLogCommand.class;
		}

	}

	@Override
	protected String[][] generateOutputTable(List<AuditLogEntry> logMailQueryResults) {
		String[][] data = new String[logMailQueryResults.size()][HEADERS.length];
		int index = 0;
		for (AuditLogEntry log : logMailQueryResults) {
			data[index][0] = log.timestamp.toString();
			data[index][1] = log.action;
			data[index][2] = (log.securityContext != null) ? log.securityContext.displayName() : "";
			data[index][3] = (log.securityContext != null) ? log.securityContext.origin() : "";
			data[index][4] = (log.container != null) ? log.container.name() : "";
			data[index][5] = (log.container != null) ? log.container.ownerElement().email() : "";
			data[index][6] = (log.container != null) ? log.container.uid() : "";
			data[index][7] = (log.content != null) ? log.content.description() : "";
			data[index][8] = (log.content != null) ? log.content.key() : "";
			data[index][9] = (log.content != null)
					? log.content.author().stream().reduce((p, e) -> p + ", " + e).orElse("")
					: "";
			if (log.content != null) {
				List<String> receivers = new ArrayList<String>(log.content.with());
				receivers.removeAll(log.content.author());
				data[index][10] = receivers.stream().reduce((p, e) -> p + ", " + e).orElse("");
			}
			data[index][11] = (log.content != null && log.content.is().size() > 1) ? log.content.is().get(0) : "";
			data[index][12] = (log.content != null && log.content.is().size() > 1) ? log.content.is().get(1) : "";
			data[index][13] = log.updatemessage;
			index++;
		}
		return data;
	}

	@Override
	protected Map<String, String> generateOutputJSON(AuditLogEntry result) {
		Map<String, String> map = new HashMap<>();
		map.put("timestamp", result.timestamp.toString());
		map.put("action", result.action);
		map.put("securitycontext_name", (result.securityContext != null) ? result.securityContext.displayName() : "");
		map.put("securitycontext_origin", (result.securityContext != null) ? result.securityContext.origin() : "");
		map.put("container_name", (result.container != null) ? result.container.name() : "");
		map.put("container_owner", (result.container != null) ? result.container.ownerElement().email() : "");
		map.put("container_uid", (result.container != null) ? result.container.uid() : "");
		map.put("event_description", (result.content != null) ? result.content.description() : "");
		map.put("event_key", (result.content != null) ? result.content.key() : "");
		map.put("event_author",
				(result.content != null) ? result.content.author().stream().reduce((p, e) -> p + ", " + e).orElse("")
						: "");
		if (result.content != null) {
			List<String> recipients = new ArrayList<String>(result.content.with());
			recipients.removeAll(result.content.author());
			map.put("event_recipients", recipients.stream().reduce((p, e) -> p + ", " + e).orElse(""));
		}
		map.put("event_start_date",
				(result.content != null && result.content.is().size() > 1) ? result.content.is().get(0) : "");
		map.put("event_end_date",
				(result.content != null && result.content.is().size() > 1) ? result.content.is().get(1) : "");
		map.put("updatemessage", result.updatemessage);

		return map;
	}

	@Override
	protected AuditLogQuery buildQuery(ItemValue<Domain> domain) {
		AuditLogQuery auditLogQuery = defaultQuery();
		if (eventName != null && !eventName.isBlank()) {
			auditLogQuery.description = eventName;
		}
		if (eventKey != null && !eventKey.isBlank()) {
			auditLogQuery.key = eventKey;
		}
		return auditLogQuery;
	}

	@Override
	protected String[] headers() {
		return HEADERS;
	}

	@Override
	protected String[] logTypes() {
		return new String[] { LOG_TYPE };
	}
}
