/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2021
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.lmtp.filter.tnef;

import java.io.ByteArrayInputStream;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import org.apache.james.mime4j.dom.Entity;
import org.apache.james.mime4j.dom.Message;
import org.apache.james.mime4j.dom.Multipart;
import org.apache.james.mime4j.dom.SingleBody;
import org.asynchttpclient.BoundRequestBuilder;
import org.asynchttpclient.DefaultAsyncHttpClient;
import org.asynchttpclient.Realm;
import org.asynchttpclient.Realm.AuthScheme;
import org.asynchttpclient.Response;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.io.ByteStreams;

import net.bluemind.core.container.api.IContainerManagement;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.model.acl.Verb;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.IServiceProvider;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.delivery.lmtp.common.LmtpEnvelope;
import net.bluemind.delivery.lmtp.common.ResolvedBox;
import net.bluemind.delivery.lmtp.filters.FilterException;
import net.bluemind.delivery.lmtp.filters.IMessageFilter;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.group.api.IGroup;
import net.bluemind.group.api.Member;
import net.bluemind.mime4j.common.AddressableEntity;
import net.bluemind.mime4j.common.Mime4JHelper;

public class TnefFilter implements IMessageFilter {

	private static final Logger logger = LoggerFactory.getLogger(TnefFilter.class);

	private DefaultAsyncHttpClient ahc;

	public TnefFilter() {
		this.ahc = new DefaultAsyncHttpClient();
	}

	@Override
	public Message filter(LmtpEnvelope env, Message message) throws FilterException {
		return MapiEndpoint.any().map(endpoint -> withEndpoint(endpoint, env, message)).orElse(null);
	}

	private Message withEndpoint(String endpoint, LmtpEnvelope env, Message message) {
		if (message.isMultipart()) {
			Multipart mp = (Multipart) message.getBody();

			List<AddressableEntity> tree = Mime4JHelper.expandTree(mp.getBodyParts());
			Optional<AddressableEntity> winmailDat = tree.stream()
					.filter(ae -> "application/ms-tnef".equalsIgnoreCase(ae.getMimeType())
							|| "winmail.dat".equalsIgnoreCase(ae.getFilename()))
					.findAny();
			return winmailDat.map(tnef -> {
				Entity decoded = (Mime4JHelper.hasEncodedHeader(message.getHeader()))
						? Mime4JHelper.decodePartIfNecessary(tnef)
						: tnef;
				Message fromTnef = rewriteWinmail(endpoint, env.getRecipients().get(0), (SingleBody) decoded.getBody());
				if (fromTnef != null) {
					// ensure the message-id from the original message is maintained
					fromTnef.getHeader().setField(message.getHeader().getField("Message-ID"));
					fromTnef.setTo(message.getTo());
					fromTnef.setCc(message.getCc());
					fromTnef.setFrom(message.getFrom());
					logger.info("MAPI endpoint has rewritten '{}' from TNEF.", fromTnef.getSubject());
				}
				return fromTnef;
			}).orElse(null);
		}
		return null;
	}

	private Message rewriteWinmail(String endpoint, ResolvedBox resolvedBox, SingleBody tnef) {

		String email = resolvedBox.entry.email;

		IDirectory dir = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM).instance(IDirectory.class,
				resolvedBox.dom.uid);
		DirEntry directoryEntry = dir.getByEmail(email);
		String uid = directoryEntry.entryUid;
		if (directoryEntry.kind == Kind.MAILSHARE) {
			uid = getMailshareWriter(dir, resolvedBox.dom.uid, uid);
			if (uid == null) {
				logger.warn("Cannot decode TNEF message, no user has permissions on target mailshare {}",
						directoryEntry.email);
				return null;
			}
		}

		logger.info("Use {} to process tnef sent to {}, body is {}", endpoint, resolvedBox, tnef.getClass());

		try (Sudo sudo = Sudo.byUid(uid, resolvedBox.dom.uid)) {
			Realm realm = new Realm.Builder(sudo.getLatd(), sudo.getAuthKey()).setScheme(AuthScheme.BASIC)
					.setCharset(StandardCharsets.UTF_8).build();
			BoundRequestBuilder post = ahc.preparePost(endpoint).setRealm(realm);
			try (InputStream in = tnef.getInputStream()) {
				byte[] read = ByteStreams.toByteArray(in);
				logger.info("Read {} byte(s) of tnef data", read.length);
				post.setBody(read);
				Response response = post.execute().get(20, TimeUnit.SECONDS);
				byte[] freshEml = response.getResponseBodyAsBytes();
				if (freshEml.length == 0) {
					Path copy = Files.createTempFile("winmail-lmtpd-unparsed", ".dat");
					Files.write(copy, read, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING,
							StandardOpenOption.WRITE);
					logger.error(
							"MAPI endpoint failed to analyse tnef, a copy was saved to {}. Mail will be delivered as-is.",
							copy.toFile().getAbsolutePath());
					return null;
				} else {
					logger.info("Re-generated EML size is {} byte(s)", freshEml.length);
					return Mime4JHelper.parse(new ByteArrayInputStream(freshEml));
				}
			} catch (InterruptedException ie) {
				Thread.currentThread().interrupt();
			} catch (Exception e) {
				logger.error(e.getMessage(), e);
			}
		}
		return null;
	}

	private String getMailshareWriter(IDirectory dir, String domainUid, String mbox) {
		IServiceProvider sp = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM);
		IContainerManagement mailboxAclsService = sp.instance(IContainerManagement.class, "mailbox:acls-" + mbox);
		List<String> writers = mailboxAclsService.getAccessControlList().stream()
				.filter(entity -> entity.verb.can(Verb.Write)).map(entry -> entry.subject).collect(Collectors.toList());
		if (writers.isEmpty()) {
			return null;
		}
		List<ItemValue<DirEntry>> entries = dir.getMultiple(writers);
		Optional<ItemValue<DirEntry>> dirEntry = entries.stream().filter(de -> de.value.kind == Kind.USER).findFirst();
		String entryUid = null;
		if (!dirEntry.isPresent()) {
			IGroup groupService = sp.instance(IGroup.class, domainUid);
			List<ItemValue<DirEntry>> groups = entries.stream().filter(de -> de.value.kind == Kind.GROUP)
					.collect(Collectors.toList());
			for (int i = 0; i < groups.size(); i++) {
				ItemValue<DirEntry> g = groups.get(i);
				List<Member> members = groupService.getExpandedUserMembers(g.uid);
				if (!members.isEmpty()) {
					entryUid = members.get(0).uid;
					break;
				}
			}
		} else {
			entryUid = dirEntry.get().uid;
		}
		return entryUid;
	}

}
