package net.bluemind.directory.hollow.datamodel.producer;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.core.container.model.ItemValue;
import net.bluemind.directory.api.BaseDirEntry;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.hollow.datamodel.AddressBookRecord;
import net.bluemind.directory.hollow.datamodel.Cert;
import net.bluemind.directory.hollow.datamodel.DataLocation;
import net.bluemind.directory.hollow.datamodel.OfflineAddressBook;
import net.bluemind.directory.hollow.datamodel.producer.EdgeNgram.EmailEdgeNGram;
import net.bluemind.domain.api.Domain;
import net.bluemind.mailbox.api.Mailbox;

public class EntryToAdressBookMapper {

	private static Logger logger = LoggerFactory.getLogger(EntryToAdressBookMapper.class);

	public static List<AddressBookRecord> mapMuliple(ItemValue<Domain> domain, List<ItemValue<DirEntry>> entries,
			Map<String, ItemValue<Mailbox>> mailboxesByUid, Map<String, DataLocation> locationCache,
			String installationId) {

		Function<ItemValue<DirEntry>, Kind> byKind = e -> e.value.kind;
		Map<Kind, List<ItemValue<DirEntry>>> dirByKind = entries.stream()
				.collect(Collectors.groupingBy(byKind, Collectors.toList()));

		Stream<Entry<Kind, List<ItemValue<DirEntry>>>> dirByKindEntrySet = dirByKind.entrySet().stream();

		return dirByKindEntrySet
				.flatMap(entrySet -> DirEntrySerializer.get(domain.uid, entrySet.getKey(), entrySet.getValue()).stream()
						.map((DirEntrySerializer serializer) -> {
							AddressBookRecord rec = new AddressBookRecord();
							ItemValue<DirEntry> entry = serializer.getDirEntry();
							ItemValue<Mailbox> mailBox = mailboxesByUid.get(entry.uid);
							return prepareRecord(domain, entry, mailBox, locationCache, installationId, rec,
									serializer);
						}))
				.toList();
	}

	public static Optional<AddressBookRecord> map(ItemValue<Domain> domain, ItemValue<DirEntry> entry,
			ItemValue<Mailbox> box, Map<String, DataLocation> datalocationCache, String installationId) {
		AddressBookRecord rec = new AddressBookRecord();
		Optional<AddressBookRecord> optRec = DirEntrySerializer.get(domain.uid, entry)
				.map((DirEntrySerializer serializer) -> prepareRecord(domain, entry, box, datalocationCache,
						installationId, rec, serializer));
		if (!optRec.isPresent()) {
			logger.warn("Integrity problem on entry {}", entry);
		}
		return optRec;
	}

	public static OfflineAddressBook createOabEntry(ItemValue<Domain> domain, long version) {
		OfflineAddressBook oab = new OfflineAddressBook();
		oab.containerGuid = UUID.nameUUIDFromBytes((domain.internalId + ":" + domain.uid).getBytes()).toString();
		oab.hierarchicalRootDepartment = null;
		oab.distinguishedName = "/";
		oab.name = "\\Default Global Address List";
		oab.sequence = (int) version;
		oab.domainName = domain.uid;
		oab.domainAliases = domain.value.aliases;
		return oab;
	}

	@SuppressWarnings("unchecked")
	private static AddressBookRecord prepareRecord(ItemValue<Domain> domain, ItemValue<DirEntry> entry,
			ItemValue<Mailbox> box, Map<String, DataLocation> datalocationCache, String installationId,
			AddressBookRecord rec, DirEntrySerializer serializer) {
		rec.uid = entry.uid;
		rec.distinguishedName = entryDN(entry.value.kind, rec.uid, domain.uid, installationId);
		rec.minimalid = entry.value.minId;
		rec.created = serializer.get(DirEntrySerializer.Property.Created).toDate();
		rec.updated = serializer.get(DirEntrySerializer.Property.Updated).toDate();
		rec.domain = domain.uid;
		String server = entry.value.dataLocation;
		if (server != null) {
			rec.dataLocation = datalocationCache.computeIfAbsent(server, s -> {
				@SuppressWarnings("unchecked")
				List<String> list = (List<String>) serializer.get(DirEntrySerializer.Property.DataLocation).toList();
				DataLocation location = null;
				if (!list.isEmpty()) {
					location = new DataLocation(list.get(0), list.get(1));
				}
				return location;
			});
		}
		Set<String> aliases = new HashSet<>();
		aliases.add(domain.uid);
		aliases.addAll(domain.value.aliases);
		rec.emails = entry.value.kind != Kind.EXTERNALUSER ? toEmails(box, aliases) : toEmails(entry.value.email);
		rec.name = serializer.get(DirEntrySerializer.Property.DisplayName).toString();
		rec.email = serializer.get(DirEntrySerializer.Property.Email).toString();
		rec.kind = serializer.get(DirEntrySerializer.Property.Kind).toString();
		rec.surname = serializer.get(DirEntrySerializer.Property.Surname).toString();
		rec.givenName = serializer.get(DirEntrySerializer.Property.GivenName).toString();
		rec.title = serializer.get(DirEntrySerializer.Property.Title).toString();
		rec.officeLocation = null;
		rec.departmentName = serializer.get(DirEntrySerializer.Property.DepartmentName).toString();
		rec.companyName = serializer.get(DirEntrySerializer.Property.CompanyName).toString();
		rec.assistant = serializer.get(DirEntrySerializer.Property.Assistant).toString();
		rec.addressBookManagerDistinguishedName = serializer
				.get(DirEntrySerializer.Property.AddressBookManagerDistinguishedName).toString();
		rec.addressBookPhoneticGivenName = serializer.get(DirEntrySerializer.Property.GivenName).toString();
		rec.addressBookPhoneticSurname = serializer.get(DirEntrySerializer.Property.Surname).toString();
		rec.addressBookPhoneticCompanyName = serializer.get(DirEntrySerializer.Property.CompanyName).toString();
		rec.addressBookPhoneticDepartmentName = serializer.get(DirEntrySerializer.Property.DepartmentName).toString();
		rec.streetAddress = serializer.get(DirEntrySerializer.Property.StreetAddress).toString();
		rec.postOfficeBox = serializer.get(DirEntrySerializer.Property.postOfficeBox).toString();
		rec.locality = serializer.get(DirEntrySerializer.Property.Locality).toString();
		rec.stateOrProvince = serializer.get(DirEntrySerializer.Property.StateOrProvince).toString();
		rec.postalCode = serializer.get(DirEntrySerializer.Property.PostalCode).toString();
		rec.country = serializer.get(DirEntrySerializer.Property.Country).toString();
		rec.businessTelephoneNumber = serializer.get(DirEntrySerializer.Property.BusinessTelephoneNumber).toString();
		rec.homeTelephoneNumber = serializer.get(DirEntrySerializer.Property.HomeTelephoneNumber).toString();
		rec.business2TelephoneNumbers = serializer.get(DirEntrySerializer.Property.Business2TelephoneNumbers)
				.toString();
		rec.mobileTelephoneNumber = serializer.get(DirEntrySerializer.Property.MobileTelephoneNumber).toString();
		rec.pagerTelephoneNumber = serializer.get(DirEntrySerializer.Property.PagerTelephoneNumber).toString();
		rec.primaryFaxNumber = serializer.get(DirEntrySerializer.Property.PrimaryFaxNumber).toString();
		rec.assistantTelephoneNumber = serializer.get(DirEntrySerializer.Property.AssistantTelephoneNumber).toString();
		rec.userCertificate = null;
		List<Value> certs = (List<Value>) serializer.get(DirEntrySerializer.Property.AddressBookX509Certificate)
				.toList();
		rec.addressBookX509Certificate = certs.stream().map(val -> new Cert(val.toByteArray())).toList();
		List<Value> certsDer = (List<Value>) serializer.get(DirEntrySerializer.Property.UserX509Certificate).toList();
		rec.userX509Certificate = certsDer.stream().map(val -> new Cert(val.toByteArray())).toList();
		rec.thumbnail = serializer.get(DirEntrySerializer.Property.ThumbnailPhoto).toByteArray();
		rec.hidden = serializer.get(DirEntrySerializer.Property.Hidden).toBoolean();
		rec.anr = new AnrTokens().compute(rec);
		return rec;
	}

	private static List<net.bluemind.directory.hollow.datamodel.Email> toEmails(ItemValue<Mailbox> box,
			Set<String> domainAliases) {
		if (box == null || box.value == null || box.value.emails == null) {
			return Collections.emptyList();
		}

		return box.value.emails.stream().flatMap(e -> {
			if (!e.allAliases) {
				return Stream.of(new net.bluemind.directory.hollow.datamodel.Email(e.address, List.of(), e.isDefault,
						e.allAliases));
			} else {
				String left = e.address.split("@")[0];
				return domainAliases.stream().map(d -> {
					String value = left + "@" + d;
					return new net.bluemind.directory.hollow.datamodel.Email(value, List.of(), e.isDefault,
							e.allAliases);
				});
			}
		}).collect(Collectors.toList());
	}

	public static String entryDN(BaseDirEntry.Kind kind, String entryUid, String domainUid, String installationId) {
		return String.format("%s/ou=%s/cn=Recipients/cn=%s:%s", getOrgDn(installationId), domainUid, kind.toString(),
				entryUid).toLowerCase();
	}

	public static String getOrgDn(String installationId) {
		return "/o=Mapi";
	}

	private static List<net.bluemind.directory.hollow.datamodel.Email> toEmails(String address) {
		return Arrays.asList(new net.bluemind.directory.hollow.datamodel.Email(address,
				new EmailEdgeNGram().compute(address), true, false));
	}
}
