/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */

package net.bluemind.maintenance.checker.hotupgrades;

import java.util.List;
import java.util.stream.Collectors;

import net.bluemind.core.rest.IServiceProvider;
import net.bluemind.core.task.api.TaskRef;
import net.bluemind.core.task.api.TaskStatus;
import net.bluemind.core.task.api.TaskStatus.State;
import net.bluemind.maintenance.checker.report.ITaskFollowingReport;
import net.bluemind.system.api.hot.upgrade.HotUpgradeTask;
import net.bluemind.system.api.hot.upgrade.HotUpgradeTaskExecutionMode;
import net.bluemind.system.api.hot.upgrade.HotUpgradeTaskFilter;
import net.bluemind.system.api.hot.upgrade.HotUpgradeTaskStatus;
import net.bluemind.system.api.hot.upgrade.IInternalHotUpgrade;

public class HotUpgradesChecker {
	private IServiceProvider provider;
	private HotUpgradeTaskFilter filter = HotUpgradeTaskFilter.filter(HotUpgradeTaskStatus.PLANNED,
			HotUpgradeTaskStatus.FAILURE);
	private boolean repair = true;

	public HotUpgradesChecker(IServiceProvider serviceProviderInfiniteQuery) {
		provider = serviceProviderInfiniteQuery;
	}

	public HotUpgradesChecker withRepair(boolean repair) {
		this.repair = repair;
		return this;
	}

	public void check(ITaskFollowingReport report) {
		IInternalHotUpgrade hotUpgradeApi = provider.instance(IInternalHotUpgrade.class);
		List<HotUpgradeTask> matches = hotUpgradeApi.list(filter).stream().filter(hu -> hu.mandatory).toList();

		if (matches.isEmpty()) {
			report.info("No pending hot upgrade");
			return;
		}

		report.warn("Hot upgrades pending: {}", matches.stream() //
				.map(u -> u.operation + ":" + u.status.toString()) //
				.collect(Collectors.joining(", ")));

		if (repair) {
			report.info("Starting scheduled hot upgrades");
			IInternalHotUpgrade service = provider.instance(IInternalHotUpgrade.class);

			TaskRef directTask = service.start(false, HotUpgradeTaskExecutionMode.DIRECT);
			TaskStatus directStatus = report.follow(directTask, "Direct", "Direct task execution failed");
			reportStatus(report, directStatus, "Direct");

			TaskRef jobTask = service.start(false, HotUpgradeTaskExecutionMode.JOB);
			TaskStatus jobStatus = report.follow(jobTask, "Job", "Job task execution failed");
			reportStatus(report, jobStatus, "Job");
		}
	}

	private void reportStatus(ITaskFollowingReport report, TaskStatus status, String upgradeType) {
		if (status != null && status.state == State.Success) {
			report.info("[{}] Tasks execution successful.", upgradeType);
		} else {
			report.error("[{}] Tasks execution unsuccessful {}:{.}", upgradeType, status.state.name(), status.result);
		}

	}
}
