/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2023
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.cli.group;

import java.util.Arrays;
import java.util.List;
import java.util.Optional;

import net.bluemind.cli.cmd.api.CliContext;
import net.bluemind.cli.cmd.api.CliException;
import net.bluemind.cli.cmd.api.ICmdLet;
import net.bluemind.cli.cmd.api.ICmdLetRegistration;
import net.bluemind.cli.utils.CliUtils;
import net.bluemind.core.container.api.ContainerQuery;
import net.bluemind.core.container.api.IContainerManagement;
import net.bluemind.core.container.api.IContainers;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.container.model.acl.AccessControlEntry;
import net.bluemind.core.container.model.acl.Verb;
import net.bluemind.domain.api.Domain;
import net.bluemind.group.api.IGroup;
import net.bluemind.group.api.Member;
import net.bluemind.group.api.Member.Type;
import picocli.CommandLine.ArgGroup;
import picocli.CommandLine.Command;
import picocli.CommandLine.Option;

@Command(name = "set-delegation-acl", description = "Manage delegation acl of a user's group (group with write acces on mailbox container)")
public class GroupSetDelegationAclCommand implements ICmdLet, Runnable {
	public static class Reg implements ICmdLetRegistration {
		@Override
		public Optional<String> group() {
			return Optional.of("group");
		}

		@Override
		public Class<? extends ICmdLet> commandClass() {
			return GroupSetDelegationAclCommand.class;
		}
	}

	private CliContext ctx;
	private CliUtils cliUtils;

	@Option(names = "--domain", required = true, description = "Target domain - must not be global.virt")
	private String domain;

	@ArgGroup(exclusive = true, heading = "All domain's groups if --name or --uid is not set\r\n")
	private GroupOptions groupOptions;

	private static class GroupOptions {
		@Option(names = "--name", required = true, description = "Target group name")
		private String name;

		@Option(names = "--uid", required = true, description = "Target group UID")
		private String uid;
	}

	@Option(names = "--verb", required = true, description = "Delegation access verb (SendOnBehalf, SendAs)")
	private String verb;

	@Option(names = "--dry", description = "Dry-run (do nothing)")
	public boolean dry;

	private static final String DRY_MODE_PREFIX = "DRY MODE: ";

	@Override
	public void run() {
		ItemValue<Domain> domainValue = cliUtils.getNotGlobalDomain(domain);

		try {
			Verb.valueOf(verb);
		} catch (Exception e) {
			throw new CliException(
					String.format("Unauthorized verb '%s', only accept [%s,%s]", verb, Verb.SendOnBehalf, Verb.SendAs));
		}

		if (dry) {
			ctx.info(DRY_MODE_PREFIX + "Only list members with mailbox WRITE access inherited from {}",
					groupOptions != null
							? "group '" + groupOptions.name != null ? groupOptions.name : groupOptions.uid + "'"
							: "all '" + domainValue.displayName + "' domain's groups");
		}

		ContainerQuery query = ContainerQuery.type("mailboxacl");
		query.verb = Arrays.asList(Verb.Write);

		IGroup groupApi = ctx.adminApi().instance(IGroup.class, domainValue.uid);
		if (groupOptions == null) {
			if (!dry) {
				ctx.info("Verify all '{}' domain's groups", domainValue.displayName);
			}
			groupApi.allUids()
					.forEach(groupUid -> singleGroupTreatment(groupApi, domainValue, query, groupUid, groupUid));
		} else {
			String groupUid = Optional.ofNullable(groupOptions.name).map(groupApi::byName).map(g -> g.uid)
					.orElse(groupOptions.uid);
			String groupLogInfo = groupOptions.name != null ? groupOptions.name : groupOptions.uid;

			if (groupUid == null) {
				throw new CliException("Group " + groupLogInfo + " not found!");
			}

			singleGroupTreatment(groupApi, domainValue, query, groupUid, groupLogInfo);
		}
	}

	private void singleGroupTreatment(IGroup groupApi, ItemValue<Domain> domainValue, ContainerQuery query,
			String groupUid, String groupLogInfo) {
		String dryModeLogPrefix = dry ? DRY_MODE_PREFIX : "";
		List<Member> members = groupApi.getMembers(groupUid);
		ctx.info(dryModeLogPrefix + "Verify all {} members WRITE access for group '{}'", members.size(), groupLogInfo);
		members.stream().filter(m -> m.type == Type.user).map(m -> m.uid).forEach(member -> {
			List<String> mailboxContainerUids = ctx.adminApi().instance(IContainers.class)
					.allForUser(domainValue.uid, member, query).stream().filter(c -> !c.owner.equalsIgnoreCase(member))
					.map(c -> c.uid).toList();

			mailboxContainerUids.stream().forEach(mailboxUid -> {
				IContainerManagement containerMgmt = ctx.adminApi().instance(IContainerManagement.class, mailboxUid);
				List<AccessControlEntry> accessControlList = containerMgmt.getAccessControlList();

				boolean groupHasWriteAccess = accessControlList.stream()
						.anyMatch(ace -> ace.verb.can(Verb.Write) && ace.subject.equalsIgnoreCase(groupUid));
				if (groupHasWriteAccess) {
					boolean hasAlreadyMaxDelegationAccess = accessControlList.stream()
							.anyMatch(ace -> Verb.valueOf(verb).can(Verb.SendOnBehalf) && ace.verb.can(Verb.SendAs)
									&& ace.subject.equalsIgnoreCase(member));
					if (hasAlreadyMaxDelegationAccess) {
						ctx.warn(dryModeLogPrefix
								+ "No '{}' access add to Member '{}' on '{}' (because the 'SendAs' access is already present)",
								verb, member, mailboxUid);
					} else {
						ctx.info(dryModeLogPrefix
								+ "Add '{}' access to Member '{}' on '{}' (because it inherits the 'WRITE' access from the group)",
								verb, member, mailboxUid);
						accessControlList.add(AccessControlEntry.create(member, Verb.valueOf(verb)));
						if (!dry) {
							containerMgmt.setAccessControlList(accessControlList);
						}
					}
				} else {
					ctx.warn(dryModeLogPrefix
							+ "No '{}' access add to Member '{}' on '{}' (because it does not inherit the 'WRITE' access from the group)",
							verb, member, mailboxUid);
				}

			});

		});
	}

	@Override
	public Runnable forContext(CliContext ctx) {
		this.ctx = ctx;
		this.cliUtils = new CliUtils(ctx);
		return this;
	}
}
