/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.common.backup;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.function.Predicate;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.type.TypeReference;

import io.netty.buffer.ByteBuf;
import io.netty.buffer.ByteBufOutputStream;
import io.netty.buffer.Unpooled;
import net.bluemind.calendar.api.internal.CalendarHistory;
import net.bluemind.core.utils.JsonUtils;
import net.bluemind.core.utils.JsonUtils.ValueReader;
import net.bluemind.core.utils.JsonUtils.ValueWriter;
import net.bluemind.dataprotect.api.IBackupWorker;
import net.bluemind.domain.api.Domain;
import net.bluemind.resource.api.ResourceDescriptor;
import net.bluemind.user.api.User;

public class JsonReadWrite {

	private static Logger logger = LoggerFactory.getLogger(JsonReadWrite.class);

	private static final TypeReference<BackupContainerItemDescriptor<User>> USER_TYPE_REFERENCE = new TypeReference<BackupContainerItemDescriptor<User>>() {
	};

	private static final TypeReference<BackupItemDescriptor<CalendarHistory>> CAL_HIST_TYPE_REFERENCE = new TypeReference<BackupItemDescriptor<CalendarHistory>>() {
	};

	private static final TypeReference<BackupContainerItemDescriptor<Domain>> DOMAIN_TYPE_REFERENCE = new TypeReference<BackupContainerItemDescriptor<Domain>>() {
	};

	private static final TypeReference<BackupContainerItemDescriptor<ResourceDescriptor>> RESOURCE_TYPE_REFERENCE = new TypeReference<BackupContainerItemDescriptor<ResourceDescriptor>>() {
	};

	private JsonReadWrite() {
	}

	public static ValueReader<BackupContainerItemDescriptor<User>> readerUser() {
		return JsonUtils.reader(USER_TYPE_REFERENCE);
	}

	public static ValueWriter writerUser() {
		return JsonUtils.writer(USER_TYPE_REFERENCE.getType());
	}

	public static ValueReader<BackupItemDescriptor<CalendarHistory>> readerCalHistory() {
		return JsonUtils.reader(CAL_HIST_TYPE_REFERENCE);
	}

	public static ValueWriter writerHistory() {
		return JsonUtils.writer(CAL_HIST_TYPE_REFERENCE.getType());
	}

	public static ValueReader<BackupContainerItemDescriptor<Domain>> readerDomain() {
		return JsonUtils.reader(DOMAIN_TYPE_REFERENCE);
	}

	public static ValueReader<BackupContainerItemDescriptor<ResourceDescriptor>> readerResource() {
		return JsonUtils.reader(RESOURCE_TYPE_REFERENCE);
	}

	public static ValueWriter writerResource() {
		return JsonUtils.writer(RESOURCE_TYPE_REFERENCE.getType());
	}

	public static void writeDomain(Path p, Object content) throws IOException {
		ValueWriter writer = JsonUtils.writer(DOMAIN_TYPE_REFERENCE.getType());
		write(writer, p, content);
	}

	public static void writeUser(Path p, Object content) throws IOException {
		ValueWriter writer = writerUser();
		write(writer, p, content);
	}

	public static void writeHistory(Path p, Object content) throws IOException {
		ValueWriter writer = JsonUtils.writer(CAL_HIST_TYPE_REFERENCE.getType());
		write(writer, p, content);
	}

	public static void writeResource(Path p, Object content) throws IOException {
		ValueWriter writer = JsonUtils.writer(RESOURCE_TYPE_REFERENCE.getType());
		write(writer, p, content);
	}

	public static void write(ValueWriter writer, Path p, Object content) throws IOException {
		Files.createDirectories(p.getParent());
		try (OutputStream out = Files.newOutputStream(p, StandardOpenOption.CREATE,
				StandardOpenOption.TRUNCATE_EXISTING)) {
			out.write(writer.write(content));
		}
	}

	public static byte[] write(ValueWriter writer, Object content) throws IOException {
		return writer.write(content);
	}

	private static boolean filterOnFilenamePattern(String filename) {
		return filename.endsWith(RepositoryBackupPath.JSON_FILE_EXTENSION)
				&& !filename.endsWith(RepositoryBackupPath.HISTORY_JSON_FILE_EXTENSION);
	}

	public static List readJsonFile(ValueReader reader, Path filePath) throws IOException {
		return read(reader, filePath, p -> filterOnFilenamePattern(p.getFileName().toString()));
	}

	public static List readJsonFiles(ValueReader reader, List<Path> filePaths) throws IOException {
		return readFiles(reader, filePaths, p -> filterOnFilenamePattern(p.getFileName().toString()));
	}

	public static List read(ValueReader reader, Path filePath, Predicate<Path> predicate) throws IOException {
		logger.info("Listing files into {}", filePath);
		if (Files.exists(filePath)) {
			try (java.util.stream.Stream<Path> s = Files.list(filePath)) {
				return s.filter(predicate).map(p -> {
					logger.info("Read file {}", p);
					try (InputStream in = Files.newInputStream(p)) {
						return reader.read(in.readAllBytes());
					} catch (IOException e) {
						logger.error("Unable to read backup json file", e);
						return Collections.emptyList();
					}
				}).filter(Objects::nonNull).toList();
			}
		} else {
			return Collections.emptyList();
		}
	}

	public static List readFiles(ValueReader reader, List<Path> filePath, Predicate<Path> predicate) {
		logger.info("Listing files into {}", filePath);
		return filePath.stream().filter(predicate).map(p -> {
			try (InputStream in = Files.newInputStream(p)) {
				return reader.read(in.readAllBytes());
			} catch (IOException e) {
				logger.error("Unable to read backup json file", e);
				return Collections.emptyList();
			}
		}).filter(Objects::nonNull).toList();
	}

	public static Object readJsonFileObj(ValueReader reader, Path filePath) throws IOException {
		if (!filePath.getFileName().toString().endsWith(RepositoryBackupPath.JSON_FILE_EXTENSION)) {
			return null;
		}
		return readObj(reader, filePath);
	}

	private static Object readObj(ValueReader reader, Path filePath) throws IOException {
		logger.info("Read file {}", filePath);
		if (Files.exists(filePath)) {
			try (InputStream in = Files.newInputStream(filePath)) {
				return reader.read(in.readAllBytes());
			} catch (IOException e) {
				logger.error("Unable to read backup json file", e);
			}
		}
		return null;
	}

	public static Object readByteArr(ValueReader reader, ByteBuf content) throws IOException {
		return reader.read(content);
	}

	public static ByteBuf readEntryData(InputStream zis) throws IOException {
		ByteBuf target = IBackupWorker.mmapOutput();
		try (OutputStream out = new ByteBufOutputStream(target)) {
			zis.transferTo(out);
		}
		return target;
	}

	public static List<ByteBuf> findJsonFilesWithPattern(String folderName, Path zipFilePath) throws IOException {
		Map<String, List<ByteBuf>> filesInZipFileWithPattern = findFilesInZipFileWithPattern(zipFilePath, folderName,
				Arrays.asList(RepositoryBackupPath.JSON_FILE_EXTENSION));
		final List<ByteBuf> list = filesInZipFileWithPattern.get(RepositoryBackupPath.JSON_FILE_EXTENSION);
		return list == null ? Collections.emptyList() : list;
	}

	private static Map<String, List<ByteBuf>> findFilesInZipFileWithPattern(Path zipFilePath, String folderName,
			List<String> fileExtensions) throws IOException {

		Map<String, List<ByteBuf>> results = new HashMap<>();

		try (ZipFile zipFile = new ZipFile(zipFilePath.toFile())) {
			Enumeration<? extends ZipEntry> entries = zipFile.entries();

			while (entries.hasMoreElements() && results.size() < 2) {
				ZipEntry entry = entries.nextElement();

				if (entry.isDirectory()) {
					continue;
				}

				String normalizedPath = entry.getName().replace("\\", "/");
				if (normalizedPath.startsWith(folderName)) {
					String filename = Path.of(normalizedPath).getFileName().toString();
					fileExtensions.forEach(ext -> {
						if (results.get(ext) == null) {
							results.put(ext, new ArrayList<>());
						}
						if (filename.endsWith(ext)
								&& !filename.endsWith(RepositoryBackupPath.HISTORY_JSON_FILE_EXTENSION)) {
							try (InputStream is = zipFile.getInputStream(entry)) {
								results.get(ext).add(readEntryData(is));
							} catch (IOException e) {
								results.get(ext).add(Unpooled.EMPTY_BUFFER);
								logger.error(e.getMessage(), e);
							}
						}
					});
				}
			}
		}

		return results;
	}

	public static ByteBuf findFilesInZipFile(Path zipFilePath, String filename) throws IOException {

		ByteBuf jsonData = Unpooled.EMPTY_BUFFER;

		try (ZipFile zipFile = new ZipFile(zipFilePath.toFile())) {
			ZipEntry jsonEntry = zipFile.getEntry(filename);
			if (jsonEntry != null) {
				try (InputStream is = zipFile.getInputStream(jsonEntry)) {
					jsonData = readEntryData(is);
				} catch (IOException e1) {
					logger.error(e1.getMessage(), e1);
				}
			}
		}

		return jsonData;
	}

	public static Map<String, ByteBuf> findFilesInZipFile(Path zipFilePath, String folderName, String adBookUid,
			List<String> fileExtensions) throws IOException {

		Map<String, ByteBuf> results = new HashMap<>();

		try (ZipFile zipFile = new ZipFile(zipFilePath.toFile())) {
			fileExtensions.forEach(ext -> {
				Path filename = Path.of(folderName, adBookUid.concat(ext));
				ZipEntry jsonEntry = zipFile.getEntry(filename.toString());
				if (jsonEntry != null) {
					try (InputStream is = zipFile.getInputStream(jsonEntry)) {
						ByteBuf jsonData = readEntryData(is);
						results.put(ext, jsonData);
					} catch (IOException e1) {
						results.put(ext, Unpooled.EMPTY_BUFFER);
						logger.error(e1.getMessage(), e1);
					}
				}
			});
		}

		return results;
	}

}
