package net.bluemind.central.reverse.proxy.stream;

import static net.bluemind.central.reverse.proxy.common.ProxyEventBusAddress.ADDRESS;
import static net.bluemind.central.reverse.proxy.common.ProxyEventBusAddress.STREAM_READY;
import static net.bluemind.central.reverse.proxy.common.config.CrpConfig.Kafka.BOOTSTRAP_SERVERS;
import static net.bluemind.central.reverse.proxy.common.config.CrpConfig.Stream.APPLICATION_ID;
import static net.bluemind.central.reverse.proxy.common.config.CrpConfig.Stream.NUMBER_OF_THREADS;
import static net.bluemind.central.reverse.proxy.common.config.CrpConfig.Topic.CLEANUP_POLICY;
import static net.bluemind.central.reverse.proxy.common.config.CrpConfig.Topic.COMPRESSION_TYPE;
import static net.bluemind.central.reverse.proxy.common.config.CrpConfig.Topic.MAX_COMPACTION_LAG_MS;
import static net.bluemind.central.reverse.proxy.common.config.CrpConfig.Topic.NAME_SUFFIX;
import static net.bluemind.central.reverse.proxy.common.config.CrpConfig.Topic.PARTITION_COUNT;
import static net.bluemind.central.reverse.proxy.common.config.CrpConfig.Topic.REPLICATION_FACTOR;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import org.apache.kafka.clients.admin.CreateTopicsOptions;
import org.apache.kafka.clients.admin.NewTopic;
import org.apache.kafka.common.serialization.Deserializer;
import org.apache.kafka.common.serialization.Serde;
import org.apache.kafka.common.serialization.Serdes;
import org.apache.kafka.common.serialization.Serdes.ByteArraySerde;
import org.apache.kafka.common.serialization.Serializer;
import org.apache.kafka.streams.KafkaStreams;
import org.apache.kafka.streams.KafkaStreams.State;
import org.apache.kafka.streams.KafkaStreams.StateListener;
import org.apache.kafka.streams.KeyValue;
import org.apache.kafka.streams.StreamsBuilder;
import org.apache.kafka.streams.StreamsConfig;
import org.apache.kafka.streams.errors.StreamsUncaughtExceptionHandler.StreamThreadExceptionResponse;
import org.apache.kafka.streams.kstream.Produced;
import org.apache.kafka.streams.processor.api.Processor;
import org.apache.kafka.streams.processor.api.ProcessorContext;
import org.apache.kafka.streams.processor.api.ProcessorSupplier;
import org.apache.kafka.streams.processor.api.Record;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.typesafe.config.Config;

import io.vertx.core.AbstractVerticle;
import io.vertx.core.Future;
import io.vertx.core.Promise;
import io.vertx.core.Vertx;
import io.vertx.core.eventbus.MessageProducer;
import io.vertx.core.json.JsonObject;
import net.bluemind.central.reverse.proxy.common.config.CrpConfig;
import net.bluemind.central.reverse.proxy.model.common.kafka.InstallationTopics;
import net.bluemind.central.reverse.proxy.model.common.kafka.KafkaAdminClient;
import net.bluemind.central.reverse.proxy.model.common.mapper.RecordKeyMapper;
import net.bluemind.central.reverse.proxy.model.common.mapper.RecordKeyMapper.PartitionnedKey;

public class DirEntriesStreamVerticle extends AbstractVerticle {

	private static final Logger logger = LoggerFactory.getLogger(DirEntriesStreamVerticle.class);

	private final Config config;
	private final String bootstrapServers;
	private final RecordKeyMapper<byte[]> keyMapper;

	private KafkaAdminClient adminClient;

	public DirEntriesStreamVerticle(Config config, RecordKeyMapper<byte[]> keyMapper) {
		this.config = config;
		this.bootstrapServers = config.getString(BOOTSTRAP_SERVERS);
		this.keyMapper = keyMapper;
	}

	@Override
	public void start(Promise<Void> p) {
		this.adminClient = KafkaAdminClient.create(bootstrapServers);

		logger.info("[stream] Starting");
		tryStart();
		p.complete();
	}

	private void tryStart() {
		try (ForestInstancesLoader loader = new ForestInstancesLoader(config)) {
			Set<String> whiteList = loader.whiteListedInstances().stream().map(s -> s.replace("-", ""))
					.collect(Collectors.toSet());
			adminClient.listTopics() //
					.map(fullSet -> onlyWhiteListed(whiteList, fullSet, config))//
					.map(topicNames -> new InstallationTopics(topicNames, config.getString(NAME_SUFFIX))) //
					.compose(this::ensureStreamOutputTopicExists) //
					.map(this::streamDirEntries) //
					.onSuccess(v -> logger.info("[stream] Started")) //
					.onFailure(t -> {
						logger.error("[stream] Failed to setup dir entries stream, retrying in 5sec", t);
						vertx.setTimer(5000, tid -> tryStart());
					});
		}
	}

	private Set<String> onlyWhiteListed(Set<String> white, Set<String> topics, Config config) {
		boolean enforce = "true".equalsIgnoreCase(System.getProperty(CrpConfig.Stream.ENFORCE_FOREST,
				Boolean.toString(config.getBoolean(CrpConfig.Stream.ENFORCE_FOREST))));
		if (enforce) {
			return topics.stream().filter(tp -> white.stream().anyMatch(tp::startsWith)).collect(Collectors.toSet());
		} else {
			return topics;
		}

	}

	private Future<InstallationTopics> ensureStreamOutputTopicExists(InstallationTopics topics) {
		if (topics.hasCrpTopic) {
			return Future.succeededFuture(topics);
		}
		NewTopic newTopic = new NewTopic(topics.crpTopicName, config.getInt(PARTITION_COUNT),
				config.getNumber(REPLICATION_FACTOR).shortValue());
		newTopic.configs(Map.of(//
				"compression.type", config.getString(COMPRESSION_TYPE), //
				"cleanup.policy", config.getString(CLEANUP_POLICY), //
				"max.compaction.lag.ms", config.getString(MAX_COMPACTION_LAG_MS)//
		));
		return adminClient.createTopic(newTopic, new CreateTopicsOptions()).map(uuid -> topics);
	}

	private static class PartitionDecorator
			implements ProcessorSupplier<byte[], byte[], PartitionnedKey<byte[]>, byte[]> {

		private final CompletableFuture<Void> firstRecordProcessed = new CompletableFuture<>();

		@Override
		public Processor<byte[], byte[], PartitionnedKey<byte[]>, byte[]> get() {
			return new Processor<byte[], byte[], RecordKeyMapper.PartitionnedKey<byte[]>, byte[]>() {
				private ProcessorContext<PartitionnedKey<byte[]>, byte[]> ctx;

				@Override
				public void init(ProcessorContext<PartitionnedKey<byte[]>, byte[]> context) {
					this.ctx = context;
				}

				@Override
				public void process(Record<byte[], byte[]> rec) {
					if (firstRecordProcessed.complete(null)) {
						logger.info("[stream] First record processed.");
					}
					ctx.recordMetadata().ifPresentOrElse(md -> {
						ctx.forward(rec.withKey(new PartitionnedKey<byte[]>(md.partition(), rec.key())));
					}, () -> DirEntriesStreamVerticle.logger.warn("No metadata for {}", rec));
				}
			};
		}

		public CompletableFuture<Void> firstRecordPromise() {
			return firstRecordProcessed;
		}

	}

	private InstallationTopics streamDirEntries(InstallationTopics topics) {
		Properties props = new Properties();
		props.put(StreamsConfig.APPLICATION_ID_CONFIG, config.getString(APPLICATION_ID));
		props.put(StreamsConfig.BOOTSTRAP_SERVERS_CONFIG, bootstrapServers);
		props.put(StreamsConfig.NUM_STREAM_THREADS_CONFIG,
				Math.min(Runtime.getRuntime().availableProcessors(), config.getInt(NUMBER_OF_THREADS)));
		props.put(StreamsConfig.DEFAULT_KEY_SERDE_CLASS_CONFIG, Serdes.ByteArraySerde.class.getName());
		props.put(StreamsConfig.DEFAULT_VALUE_SERDE_CLASS_CONFIG, Serdes.ByteArraySerde.class.getName());

		Collection<String> inputTopicNames = topics.domainTopics.values();
		String ouputTopicName = topics.crpTopicName;

		PartitionDecorator deco = new PartitionDecorator();

		StreamsBuilder topology = new StreamsBuilder();
		topology //
				.<byte[], byte[]>stream(inputTopicNames) //
				.filter((key, value) -> keyMapper.map(key)
						.map(recordKey -> value != null
								&& (recordKey.type().equals("dir") || recordKey.type().equals("memberships")))
						.orElse(false))
				.process(deco)//
				.flatMap((key, value) -> {
					Collection<KeyValue<PartitionnedKey<byte[]>, byte[]>> keyValueList = new ArrayList<>(3);
					keyMapper.map(key) //
							.filter(recordKey -> recordKey.operation().equals("DELETE")) //
							.ifPresent(recordKey -> {
								keyMapper.map(recordKey.withOperation("UPDATE"))
										.map(keyAsByteArray -> new KeyValue<PartitionnedKey<byte[]>, byte[]>(
												new PartitionnedKey<>(key.part(), keyAsByteArray), null))
										.ifPresent(keyValueList::add);
								keyMapper.map(recordKey.withOperation("CREATE"))
										.map(keyAsByteArray -> new KeyValue<PartitionnedKey<byte[]>, byte[]>(
												new PartitionnedKey<>(key.part(), keyAsByteArray), null))
										.ifPresent(keyValueList::add);
							});
					keyValueList.add(new KeyValue<>(key, value));
					return keyValueList;
				}).to(ouputTopicName, withProducer());

		KafkaStreams stream = new KafkaStreams(topology.build(), props);
		StreamMonitor mon = new StreamMonitor(stream, vertx);
		stream.setStateListener(mon);
		stream.setUncaughtExceptionHandler((Throwable throwable) -> {
			logger.error("[stream] Exception occurred during stream processing", throwable);
			stream.close();
			return StreamThreadExceptionResponse.REPLACE_THREAD;
		});
		stream.start();
		Runtime.getRuntime().addShutdownHook(new Thread(() -> {
			stream.close(Duration.ofSeconds(30));
		}));
		return publishTopics(deco, topics);
	}

	private static class StreamMonitor implements StateListener {
		private final Vertx vx;
		private final KafkaStreams stream;
		private long periodic;
		private final MessageProducer<String> statePub;
		private final MessageProducer<Long> lagPub;

		public StreamMonitor(KafkaStreams stream, Vertx vx) {
			this.stream = stream;
			this.vx = vx;
			this.statePub = vx.eventBus().publisher("dir.stream.state");
			this.lagPub = vx.eventBus().publisher("dir.stream.lag");
		}

		@Override
		public void onChange(State newState, State oldState) {
			statePub.write(newState.name());
			if (newState == KafkaStreams.State.RUNNING) {
				periodic = vx.setPeriodic(5000, tid -> {
					ConcurrentHashMap<String, Long> lags = new ConcurrentHashMap<>();
					stream.metadataForLocalThreads().forEach(tm -> {
						for (var at : tm.activeTasks()) {
							var commits = at.committedOffsets();
							var ends = at.endOffsets();
							commits.forEach((tp, commit) -> {
								long endOfset = ends.getOrDefault(tp, commit);
								long lag = endOfset - commit;
								lag = lag < 0 ? 0 : lag;
								String k = at.taskId() + "-" + tp.topic() + "#" + tp.partition();
								lags.put(k, lag);
							});
						}
					});
					long globalLag = lags.reduceValuesToLong(8, Long::longValue, 0L, (l1, l2) -> l1 + l2);
					lagPub.write(globalLag);
				});
			} else if (oldState == State.RUNNING) {
				vx.cancelTimer(periodic);
			}
		}
	}

	private static class PKSerDes implements Serde<PartitionnedKey<byte[]>> {

		@Override
		public Serializer<PartitionnedKey<byte[]>> serializer() {
			return (String topic, PartitionnedKey<byte[]> data) -> data.key();
		}

		@Override
		public Deserializer<PartitionnedKey<byte[]>> deserializer() {
			return (String topic, byte[] data) -> new PartitionnedKey<>(-1, data);
		}

	}

	private Produced<PartitionnedKey<byte[]>, byte[]> withProducer() {
		return Produced.with(new PKSerDes(), new ByteArraySerde(), //
				(String topic, PartitionnedKey<byte[]> key, byte[] value, int numPart) -> key.part() % numPart);
	}

	private InstallationTopics publishTopics(PartitionDecorator dec, InstallationTopics installationTopics) {
		Duration dur = Duration.ofSeconds(30);
		logger.info("[stream] Waiting at most {} to notify model verticle", dur);
		dec.firstRecordPromise().orTimeout(dur.toMillis(), TimeUnit.MILLISECONDS).whenComplete((v, ex) -> {
			if (ex == null) {
				logger.info("[stream] Announcing dir entries stream ready: {}", JsonObject.mapFrom(installationTopics));
			} else {
				logger.warn("[stream] Took more than {} to process the first record, we will proceed anyway", dur);
			}
			// the consumers of this message might not be ready when crp starts
			// we notify in every case as the input topic of the stream may be empty
			vertx.eventBus().publish(ADDRESS, JsonObject.mapFrom(installationTopics), STREAM_READY);
		});
		return installationTopics;
	}
}
