/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.tx.outbox.persistence;

import java.sql.SQLException;
import java.util.Collections;
import java.util.List;
import java.util.function.Supplier;

import javax.sql.DataSource;

import net.bluemind.core.api.DataSourceType;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.DataLocation;
import net.bluemind.core.rest.BmContext;
import net.bluemind.repository.provider.IDataLocationBoundFactory;
import net.bluemind.tx.outbox.repository.FlushSeqHolder;
import net.bluemind.tx.outbox.repository.ITxOutboxRepository;

public class PgRepoFactory implements IDataLocationBoundFactory<ITxOutboxRepository> {

	@Override
	public Class<ITxOutboxRepository> factoryClass() {
		return ITxOutboxRepository.class;
	}

	@Override
	public DataSourceType targetRepositoryType() {
		return DataSourceType.POSTGRESQL;
	}

	@Override
	public ITxOutboxRepository instance(BmContext context, DataLocation dataLocation) throws ServerFault {
		DataSource ds = dataLocation.equals(DataLocation.directory()) ? context.getDataSource()
				: context.getMailboxDataSource(dataLocation.serverUid());
		if (ds == null) {
			return new ITxOutboxRepository() {
				private final Supplier<FlushSeqHolder> seqHolder = () -> FlushSeqHolder.get("");

				@Override
				public long forKafka(String tgtTopic, String partKey, byte[] key, byte[] value) throws SQLException {
					return -1L;
				}

				@Override
				public List<Long> kafkaPending(int limit) throws SQLException {
					return Collections.emptyList();
				}

				@Override
				public KafkaPayload get(long id) throws SQLException {
					return null;
				}

				@Override
				public void deleteOffsets(List<Long> id) throws SQLException {
					// noop impl
				}

				@Override
				public FlushSeqHolder seqHolder() {
					return seqHolder.get();
				}
			};
		}
		return new TxOutboxStore(ds);
	}

}
