/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.eas.backend.bm.mail;

import java.util.ArrayList;
import java.util.Optional;

import com.google.common.base.Strings;

import net.bluemind.backend.mail.api.IMailboxFolders;
import net.bluemind.backend.mail.api.MailboxFolderSearchQuery;
import net.bluemind.backend.mail.api.MessageSearchResult.Mbox;
import net.bluemind.backend.mail.api.SearchQuery;
import net.bluemind.backend.mail.api.SearchResult;
import net.bluemind.backend.mail.api.SearchSort;
import net.bluemind.backend.mail.replica.api.IMailReplicaUids;
import net.bluemind.core.container.api.ContainerHierarchyNode;
import net.bluemind.core.container.api.IContainersFlatHierarchy;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.eas.backend.MailFolder;
import net.bluemind.eas.backend.dto.CollectionIdContext;
import net.bluemind.eas.dto.base.CollectionItem;
import net.bluemind.eas.dto.base.Range;
import net.bluemind.eas.dto.email.Importance;
import net.bluemind.eas.dto.find.FindRequest;
import net.bluemind.eas.dto.find.FindRequest.Options;
import net.bluemind.eas.dto.find.FindRequest.Query;
import net.bluemind.eas.dto.find.FindResponse;
import net.bluemind.eas.dto.find.FindResponse.Response;
import net.bluemind.eas.dto.find.FindResponse.Response.Result.Properties;
import net.bluemind.eas.dto.find.FindResponse.Status;
import net.bluemind.eas.dto.sync.CollectionId;
import net.bluemind.eas.exception.CollectionNotFoundException;
import net.bluemind.eas.session.BackendSession;
import net.bluemind.eas.store.ISyncStorage;

public class FindMail {

	private final BackendSession bs;
	private final FindRequest query;

	public FindMail(BackendSession bs, FindRequest query) {
		this.bs = bs;
		this.query = query;
	}

	Response search(ISyncStorage storage, IMailboxFolders mailboxFolderService, IContainersFlatHierarchy flatH)
			throws CollectionNotFoundException {
		Query searchQuery = query.executeSearch.mailBoxSearchCriterion.query;
		Options options = query.executeSearch.mailBoxSearchCriterion.options;
		FindResponse.Response response = new FindResponse.Response();
		Optional<MailFolder> folder = Optional.empty();

		if (!Strings.isNullOrEmpty(searchQuery.collectionId)) {
			folder = Optional
					.of(storage.getMailFolder(new CollectionIdContext(bs, CollectionId.of(searchQuery.collectionId))));
		}

		MailboxFolderSearchQuery mailboxSearchQuery = new MailboxFolderSearchQuery();
		mailboxSearchQuery.query = new SearchQuery();
		mailboxSearchQuery.sort = SearchSort.byField("date", SearchSort.Order.Desc);
		mailboxSearchQuery.query.recordQuery = "-is:deleted";
		mailboxSearchQuery.query.query = searchQuery.freeText;
		mailboxSearchQuery.query.scope = new net.bluemind.backend.mail.api.SearchQuery.SearchScope();
		if (folder.isPresent()) {
			mailboxSearchQuery.query.scope.folderScope = new net.bluemind.backend.mail.api.SearchQuery.FolderScope();
			mailboxSearchQuery.query.scope.folderScope.folderUid = folder.get().uid;
			if (options != null) {
				mailboxSearchQuery.query.scope.isDeepTraversal = options.deepTraversal;
			}
		}

		if (options != null && options.range != null) {
			mailboxSearchQuery.query.offset = options.range.min;
			mailboxSearchQuery.query.maxResults = options.range.max;
		}

		SearchResult results = mailboxFolderService.searchItems(mailboxSearchQuery);

		response.results = new ArrayList<>(results.results.size());
		results.results.forEach(messageSearchResult -> {
			FindResponse.Response.Result result = new FindResponse.Response.Result();

			String collectionId = searchQuery.collectionId;
			if (Strings.isNullOrEmpty(collectionId)) {
				String nodeUid = ContainerHierarchyNode.uidFor(messageSearchResult.containerUid,
						IMailReplicaUids.MAILBOX_RECORDS, bs.getUser().getDomain());
				ItemValue<ContainerHierarchyNode> hierarchyNode = flatH.getComplete(nodeUid);
				collectionId = Long.toString(hierarchyNode.internalId);
			}

			result.serverId = CollectionItem.of(collectionId, (long) messageSearchResult.itemId).toString();
			result.collectionId = collectionId;

			Properties properties = new FindResponse.Response.Result.Properties();
			properties.subject = messageSearchResult.subject;
			properties.dateReceived = messageSearchResult.date;
			properties.displayTo = mboxToString(messageSearchResult.to);
			properties.importance = messageSearchResult.flagged ? Importance.HIGH : Importance.NORMAL;
			properties.read = messageSearchResult.seen;
			properties.preview = truncate(messageSearchResult.preview, 255);
			properties.from = mboxToString(messageSearchResult.from);
			result.properties = properties;

			response.results.add(result);
		});
		response.total = results.totalResults;
		response.range = Range.create(options.range.min, (options.range.max + response.total.intValue() - 1));

		return response;
	}

	private static String mboxToString(Mbox mbox) {
		if (!Strings.isNullOrEmpty(mbox.displayName)) {
			return "\"" + mbox.displayName + "\" <" + mbox.address + ">";
		}
		return mbox.address;
	}

	private static String truncate(String text, int length) {
		if (text.length() <= length) {
			return text;
		} else {
			return text.substring(0, length);
		}
	}

	public static Response invalidRequest() {
		FindResponse.Response response = new FindResponse.Response();
		response.status = Status.INVALID_REQUEST;
		return response;

	}

}
