/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.eas.backend.bm;

import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.eas.backend.HierarchyNode;
import net.bluemind.eas.backend.importer.HierarchyImportEntity;
import net.bluemind.eas.backend.importer.HierarchyImportFolderUpdateEntity;
import net.bluemind.eas.backend.importer.IHierarchyImporter;
import net.bluemind.eas.dto.sync.CollectionId;
import net.bluemind.eas.dto.type.ItemDataType;
import net.bluemind.eas.exception.ActiveSyncException;
import net.bluemind.eas.session.BackendSession;
import net.bluemind.eas.utils.EasLogUser;

public class HierarchyImporter implements IHierarchyImporter {

	private static final Logger logger = LoggerFactory.getLogger(HierarchyImporter.class);

	private FolderBackend folderBackend;

	public HierarchyImporter(FolderBackend folderBackend) {
		this.folderBackend = folderBackend;
	}

	@Override
	public CollectionId importFolderCreate(HierarchyImportFolderUpdateEntity hierarchyEntity) {
		CollectionId collectionId = null;
		switch (hierarchyEntity.type) {
		case CALENDAR:
			collectionId = folderBackend.createFolder(hierarchyEntity);
			break;
		case CONTACTS:
			EasLogUser.logInfoAsUser(hierarchyEntity.user, logger, "Create contacts folder is not implemented");
			break;
		case EMAIL:
			collectionId = folderBackend.createMailFolder(hierarchyEntity);
			break;
		case TASKS:
			collectionId = folderBackend.createFolder(hierarchyEntity);
			break;
		default:
			break;
		}
		return collectionId;
	}

	@Override
	public boolean importFolderDelete(HierarchyImportEntity hierarchyEntity) {
		HierarchyNode node = null;
		boolean ret = false;
		try {
			node = folderBackend.getHierarchyNode(hierarchyEntity.collectionIdContext);
		} catch (ActiveSyncException e) {
			EasLogUser.logExceptionAsUser(hierarchyEntity.user, e, logger);
		}
		if (node != null) {
			HierarchyImportFolderUpdateEntity hierarchyEntityUpdate = (HierarchyImportFolderUpdateEntity) hierarchyEntity;
			hierarchyEntityUpdate.node = node;
			hierarchyEntityUpdate.type = ItemDataType.getValue(node.containerType);
			switch (hierarchyEntityUpdate.type) {
			case CALENDAR:
				ret = folderBackend.deleteFolder(hierarchyEntityUpdate);
				break;
			case CONTACTS:
				EasLogUser.logInfoAsUser(hierarchyEntityUpdate.user, logger,
						"Delete contacts folder is not implemented");
				break;
			case EMAIL:
				ret = folderBackend.deleteMailFolder(hierarchyEntityUpdate);
				break;
			case TASKS:
				ret = folderBackend.deleteFolder(hierarchyEntityUpdate);
				break;
			default:
				break;
			}
		}
		return ret;
	}

	@Override
	public boolean importFolderUpdate(HierarchyImportEntity hierarchyEntity) {
		HierarchyNode node = null;
		boolean ret = false;
		try {
			node = folderBackend.getHierarchyNode(hierarchyEntity.collectionIdContext);
		} catch (ActiveSyncException e) {
			EasLogUser.logExceptionAsUser(hierarchyEntity.user, e, logger);
		}

		if (node != null) {
			HierarchyImportFolderUpdateEntity hierarchyEntityUpdate = (HierarchyImportFolderUpdateEntity) hierarchyEntity;
			hierarchyEntityUpdate.node = node;
			hierarchyEntityUpdate.type = ItemDataType.getValue(node.containerType);
			switch (hierarchyEntityUpdate.type) {
			case CALENDAR:
				ret = folderBackend.updateFolder(hierarchyEntityUpdate);
				break;
			case CONTACTS:
				EasLogUser.logInfoAsUser(hierarchyEntityUpdate.user, logger,
						"Update contacts folder is not implemented");
				break;
			case EMAIL:
				ret = folderBackend.updateMailFolder(hierarchyEntityUpdate);
				break;
			case TASKS:
				ret = folderBackend.updateFolder(hierarchyEntityUpdate);
				break;
			default:
				break;
			}
		}
		return ret;
	}

	@Override
	public Set<String> importUserSubscriptionOwners(BackendSession bs) {
		return folderBackend.getSubscriptionOwners(bs);
	}

}
