/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.eas.backend.bm;

import java.util.Date;
import java.util.List;

import com.google.common.io.ByteSource;

import net.bluemind.eas.backend.HierarchyNode;
import net.bluemind.eas.backend.SendMailData;
import net.bluemind.eas.backend.bm.calendar.CalendarBackend;
import net.bluemind.eas.backend.bm.contacts.ContactsBackend;
import net.bluemind.eas.backend.bm.mail.MailBackend;
import net.bluemind.eas.backend.bm.task.TaskBackend;
import net.bluemind.eas.backend.dto.CollectionIdContext;
import net.bluemind.eas.backend.importer.ContentImportEntityForChange;
import net.bluemind.eas.backend.importer.ContentImportEntityForDeletion;
import net.bluemind.eas.backend.importer.ContentImportEntityForMove;
import net.bluemind.eas.backend.importer.IContentsImporter;
import net.bluemind.eas.data.calendarenum.AttendeeStatus;
import net.bluemind.eas.dto.base.CollectionItem;
import net.bluemind.eas.dto.moveitems.MoveItemsResponse;
import net.bluemind.eas.dto.type.ItemDataType;
import net.bluemind.eas.exception.ActiveSyncException;
import net.bluemind.eas.exception.CollectionNotFoundException;
import net.bluemind.eas.exception.NotAllowedException;
import net.bluemind.eas.session.BackendSession;

/**
 * 
 * 
 */
public class ContentsImporter implements IContentsImporter {

	private MailBackend mailBackend;
	private CalendarBackend calBackend;
	private ContactsBackend contactBackend;
	private TaskBackend taskBackend;

	public ContentsImporter(MailBackend mailBackend, CalendarBackend calBackend, ContactsBackend contactBackend,
			TaskBackend taskBackend) {
		this.mailBackend = mailBackend;
		this.calBackend = calBackend;
		this.contactBackend = contactBackend;
		this.taskBackend = taskBackend;
	}

	@Override
	public CollectionItem importMessageChange(ContentImportEntityForChange contentEntity) throws ActiveSyncException {
		CollectionItem ids = null;
		switch (contentEntity.type) {
		case CALENDAR:
			ids = calBackend.store(contentEntity);
			break;
		case CONTACTS:
			ids = contactBackend.store(contentEntity);
			break;
		case EMAIL:
			ids = mailBackend.store(contentEntity);
			break;
		case TASKS:
			ids = taskBackend.store(contentEntity);
			break;
		default:
			break;
		}
		return ids;
	}

	@Override
	public void importMessageDeletion(ContentImportEntityForDeletion contentEntity) throws ActiveSyncException {
		switch (contentEntity.type) {
		case CALENDAR:
			calBackend.delete(contentEntity);
			break;
		case CONTACTS:
			contactBackend.delete(contentEntity);
			break;
		case EMAIL:
			mailBackend.delete(contentEntity);
			break;
		case TASKS:
			taskBackend.delete(contentEntity);
			break;
		default:
			break;
		}
	}

	@Override
	public List<MoveItemsResponse.Response> importMoveItems(ContentImportEntityForMove contentEntity)
			throws ActiveSyncException {
		switch (contentEntity.type) {
		case CALENDAR:
			return calBackend.move(contentEntity);
		case CONTACTS:
			break;
		case EMAIL:
			return mailBackend.move(contentEntity);
		case TASKS:
			break;
		default:
			break;
		}

		return null;
	}

	@Override
	public void sendEmail(SendMailData mail) throws ActiveSyncException {
		mailBackend.sendEmail(mail);
	}

	@Override
	public void sendDraft(ContentImportEntityForChange contentEntity) throws ActiveSyncException {
		mailBackend.sendDraft(contentEntity);
	}

	@Override
	public void replyEmail(CollectionIdContext collectionIdContext, ByteSource mailContent, Boolean saveInSent,
			String serverId, boolean includePrevious) throws Exception {
		mailBackend.replyToEmail(collectionIdContext, mailContent, saveInSent, serverId, includePrevious);
	}

	@Override
	public String importCalendarUserStatus(BackendSession bs, long itemId, AttendeeStatus userResponse, Date instanceId,
			String calendarUid) {
		return calBackend.updateUserStatus(bs, itemId, userResponse, instanceId, calendarUid);
	}

	@Override
	public void forwardEmail(CollectionIdContext collectionIdContext, ByteSource mailContent, Boolean saveInSent,
			String serverId, boolean includePrevious) throws Exception {
		mailBackend.forwardEmail(collectionIdContext, mailContent, saveInSent, serverId, includePrevious);
	}

	@Override
	public void emptyFolderContent(CollectionIdContext collectionIdContext, HierarchyNode node, boolean deleteSubFolder)
			throws CollectionNotFoundException, NotAllowedException {
		if (ItemDataType.getValue(node.containerType) == ItemDataType.EMAIL) {
			mailBackend.purgeFolder(collectionIdContext, deleteSubFolder);
		} else {
			throw new NotAllowedException("emptyFolderContent is only supported for emails, collection was "
					+ ItemDataType.getValue(node.containerType));
		}

	}

}
