/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.eas.backend.bm;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Base64;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import com.google.common.annotations.VisibleForTesting;

import net.bluemind.addressbook.api.CertInfo;
import net.bluemind.addressbook.api.IAddressBooks;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.eas.backend.Changes;
import net.bluemind.eas.backend.IContentsExporter;
import net.bluemind.eas.backend.MSAttachementData;
import net.bluemind.eas.backend.bm.calendar.CalendarBackend;
import net.bluemind.eas.backend.bm.contacts.ContactsBackend;
import net.bluemind.eas.backend.bm.impl.CoreConnect;
import net.bluemind.eas.backend.bm.mail.AttachmentHelper;
import net.bluemind.eas.backend.bm.mail.MailBackend;
import net.bluemind.eas.backend.bm.task.TaskBackend;
import net.bluemind.eas.backend.dto.CollectionIdContext;
import net.bluemind.eas.dto.base.AppData;
import net.bluemind.eas.dto.base.BodyOptions;
import net.bluemind.eas.dto.base.Picture;
import net.bluemind.eas.dto.email.AttachmentResponse;
import net.bluemind.eas.dto.find.FindRequest;
import net.bluemind.eas.dto.find.FindResponse;
import net.bluemind.eas.dto.find.FindResponse.Response;
import net.bluemind.eas.dto.find.FindResponse.Status;
import net.bluemind.eas.dto.resolverecipients.ResolveRecipientsRequest;
import net.bluemind.eas.dto.resolverecipients.ResolveRecipientsRequest.Options.CertificateRetrieval;
import net.bluemind.eas.dto.resolverecipients.ResolveRecipientsResponse;
import net.bluemind.eas.dto.resolverecipients.ResolveRecipientsResponse.Response.Recipient;
import net.bluemind.eas.dto.resolverecipients.ResolveRecipientsResponse.Response.Recipient.Availability;
import net.bluemind.eas.dto.resolverecipients.ResolveRecipientsResponse.Response.Recipient.Type;
import net.bluemind.eas.dto.sync.CollectionSyncRequest.Options;
import net.bluemind.eas.dto.sync.FilterType;
import net.bluemind.eas.dto.sync.SyncState;
import net.bluemind.eas.dto.type.ItemDataType;
import net.bluemind.eas.exception.ActiveSyncException;
import net.bluemind.eas.exception.CollectionNotFoundException;
import net.bluemind.eas.exception.ObjectNotFoundException;
import net.bluemind.eas.session.BackendSession;
import net.bluemind.eas.session.ItemChangeReference;
import net.bluemind.eas.utils.EasLogUser;

public class ContentsExporter extends CoreConnect implements IContentsExporter {

	private MailBackend mailBackend;
	private CalendarBackend calBackend;
	private ContactsBackend contactsBackend;
	private TaskBackend taskBackend;
	private Map<String, FilterType> filterTypeCache;

	public ContentsExporter(MailBackend mailBackend, CalendarBackend calendarExporter, ContactsBackend contactsBackend,
			TaskBackend taskBackend) {
		this.mailBackend = mailBackend;
		this.calBackend = calendarExporter;
		this.contactsBackend = contactsBackend;
		this.taskBackend = taskBackend;
		filterTypeCache = new ConcurrentHashMap<>();
	}

	public record FilterTypeStatus(boolean hasFilterTypeChanged, Instant previousFilterTypeDate,
			Instant currentFilterTypeDate) {

	}

	@VisibleForTesting
	public FilterTypeStatus processFilterType(CollectionIdContext collectionIdContext, SyncState state,
			FilterType filterType) {

		String key = collectionIdContext.backendSession().getUniqueIdentifier() + "-"
				+ collectionIdContext.collectionId().getFolderId();
		if (filterType == null) {
			filterType = FilterType.ALL_ITEMS;
		}

		FilterType previousFilterType = filterTypeCache.getOrDefault(key, null);

		filterTypeCache.put(key, filterType);
		if (previousFilterType == null) {
			return new FilterTypeStatus(false, filterType.filterDate(), filterType.filterDate());
		}
		return new FilterTypeStatus(previousFilterType != filterType, previousFilterType.filterDate(),
				filterType.filterDate());

	}

	@Override
	public Changes getChanged(CollectionIdContext collectionIdContext, SyncState state, Options options)
			throws ActiveSyncException {

		Changes changes = new Changes();
		switch (state.type) {
		case CALENDAR:
			changes = calBackend.getContentChanges(collectionIdContext, state.version);
			break;
		case CONTACTS:
			changes = contactsBackend.getContentChanges(collectionIdContext, state.version);
			break;
		case EMAIL:
			FilterTypeStatus filterTypeStatus = processFilterType(collectionIdContext, state, options.filterType);
			changes = mailBackend.getContentChanges(collectionIdContext, state, options, filterTypeStatus);
			break;
		case TASKS:
			changes = taskBackend.getContentChanges(collectionIdContext, state.version);
			break;
		default:
			break;
		}

		EasLogUser.logInfoAsUser(collectionIdContext.getUserLogin(), logger,
				"Get changes from version {} collectionId {}, type {}, changes {}. New version {}", state.version,
				collectionIdContext.collectionId(), state.type, changes.items.size(), changes.version);

		return changes;
	}

	@Override
	public AppData loadStructure(BackendSession bs, BodyOptions bodyOptions, ItemChangeReference ic)
			throws ActiveSyncException {
		switch (ic.getType()) {
		case CALENDAR:
			return calBackend.fetch(bs, ic);
		case CONTACTS:
			return contactsBackend.fetch(bs, ic);
		case EMAIL:
			return mailBackend.fetch(bs, bodyOptions, ic);
		case TASKS:
			return taskBackend.fetch(bs, ic);
		default:
			throw new ActiveSyncException("Unsupported dataType " + ic.getType());
		}
	}

	@Override
	public Map<Long, AppData> loadStructures(CollectionIdContext collectionIdContext, BodyOptions bodyOptions,
			ItemDataType type, List<Long> ids) throws ActiveSyncException {
		switch (type) {
		case CALENDAR:
			return calBackend.fetchMultiple(collectionIdContext, ids);
		case CONTACTS:
			return contactsBackend.fetchMultiple(collectionIdContext, ids);
		case TASKS:
			return taskBackend.fetchMultiple(collectionIdContext, ids);
		case EMAIL:
			return mailBackend.fetchMultiple(collectionIdContext, bodyOptions, ids);
		default:
			throw new ActiveSyncException("Unsupported dataType " + type);

		}
	}

	@Override
	public MSAttachementData getAttachment(BackendSession bs, String attachmentId) throws ActiveSyncException {

		Map<String, String> parsedAttId = AttachmentHelper.parseAttachmentId(attachmentId);
		String type = parsedAttId.get(AttachmentHelper.TYPE);

		if (AttachmentHelper.BM_FILEHOSTING_EVENT.equals(type)) {
			return calBackend.getAttachment(bs, parsedAttId);
		} else {
			return mailBackend.getAttachment(bs, attachmentId);
		}

	}

	@Override
	public AttachmentResponse getAttachmentMetadata(BackendSession bs, String attachmentId)
			throws ObjectNotFoundException {
		if (attachmentId != null && !attachmentId.isEmpty()) {
			Map<String, String> parsedAttId = AttachmentHelper.parseAttachmentId(attachmentId);
			if (parsedAttId != null) {
				String contentType = parsedAttId.get(AttachmentHelper.CONTENT_TYPE);
				AttachmentResponse ar = new AttachmentResponse();
				ar.contentType = contentType;
				return ar;
			}
		}
		throw new ObjectNotFoundException();
	}

	@Override
	public List<Recipient> resolveRecipients(BackendSession bs, List<String> emails,
			ResolveRecipientsRequest.Options.Picture picture,
			ResolveRecipientsRequest.Options.CertificateRetrieval cert) {
		IDirectory dir = getService(bs, IDirectory.class, bs.getUser().getDomain());
		IAddressBooks allBooks = getService(bs, IAddressBooks.class);
		List<Recipient> ret = new ArrayList<>(emails.size());
		try {
			for (String to : emails) {
				Recipient recip = new Recipient();
				recip.to = to;
				DirEntry dirEntry = dir.getByEmail(to);
				if (dirEntry != null) {
					recip.emailAddress = to;
					recip.type = Type.GAL;
					recip.displayName = dirEntry.displayName;
					recip.entryUid = dirEntry.entryUid;
					if (picture != null) {
						recip.picture = Picture.noPhoto();
					}
				} else {
					recip.emailAddress = to;
					recip.type = Type.CONTACT;
					recip.displayName = to;
					if (picture != null) {
						recip.picture = Picture.noPhoto();
					}
				}

				if (cert == CertificateRetrieval.RetrieveFullCertificate) {
					// NINE app only wants one cert per user
					allBooks.findCertsByEmail(to).stream()
							.filter(ci -> ci.usage.contains(CertInfo.CertUsage.keyEncipherment)
									|| ci.usage.contains(CertInfo.CertUsage.dataEncipherment))
							.findFirst().ifPresentOrElse(ci -> {
								recip.certificate = new ResolveRecipientsResponse.Response.Recipient.Certificate();
								recip.certificate.status = ResolveRecipientsResponse.Response.Recipient.Certificate.Status.SUCCESS;
								recip.certificate.certificate = Base64.getEncoder().encodeToString(ci.x509Certificate);
								recip.certificate.certificateCount = 1;
								recip.certificate.recipientCount = 1;
							}, () -> {
								recip.certificate = new ResolveRecipientsResponse.Response.Recipient.Certificate();
								recip.certificate.status = ResolveRecipientsResponse.Response.Recipient.Certificate.Status.INVALID_SMIME_CERTIFICATE;
							});
				}

				ret.add(recip);
			}
		} catch (Exception e) {
			EasLogUser.logExceptionAsUser(bs.getLoginAtDomain(), e, logger);
		}

		return ret;
	}

	@Override
	public Availability fetchAvailability(BackendSession bs, String entryUid, Date startTime, Date endTime) {
		return calBackend.fetchAvailability(bs, entryUid, startTime, endTime);
	}

	@Override
	public Response find(BackendSession bs, FindRequest query) throws CollectionNotFoundException {
		if (query.executeSearch.mailBoxSearchCriterion != null) {
			return mailBackend.find(bs, query);
		}

		if (query.executeSearch.galSearchCriterion != null) {
			// Don't know how to trigger a Find + GALSearchCriterion
			EasLogUser.logErrorAsUser(bs.getLoginAtDomain(), logger, "[{}] GALSearchCriterion not supported",
					bs.getUniqueIdentifier());
		}

		FindResponse.Response response = new FindResponse.Response();
		response.status = Status.INVALID_REQUEST;
		return response;

	}

}
