/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.eas.config.global;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class GlobalConfig {
	private static final Logger logger = LoggerFactory.getLogger(GlobalConfig.class);

	public static final String ROOT = "/etc/bm-eas/";
	public static int EAS_PORT = 8082;
	public static String DATA_IN_LOG_FILENAME = "data.in.logs";
	public static String VALIDATE_REQUESTS_FILENAME = "validate.requests";
	public static String DISABLE_POLICIES_FILENAME = "disable.policies";

	public boolean DISABLE_POLICIES = false;
	public boolean DATA_IN_LOGS = false;
	public boolean FAIL_ON_INVALID_REQUESTS = false;
	public Set<String> USERS_WATCHED = new HashSet<>();

	private long PREVIOUS_DATA_IN_LOGS_SIZE = -1;

	private static final GlobalConfig INSTANCE = new GlobalConfig();

	private GlobalConfig() {

	}

	public static GlobalConfig get() {
		return INSTANCE;
	}

	public void loadConfig() throws IOException {
		configUpdate();
		PREVIOUS_DATA_IN_LOGS_SIZE = DATA_IN_LOGS ? getFile(DATA_IN_LOG_FILENAME).length() : -1;
	}

	public void configUpdate() throws IOException {
		disablePolicies();
		failOnInvalidRequests();
		dataInLogs();
		updateUsersListToWatch();
		if (DATA_IN_LOGS) {
			boolean all = USERS_WATCHED.isEmpty();
			logger.info("Debug content logs activated for {}", all ? "all eas users"
					: String.format("users : [%s]", USERS_WATCHED.stream().collect(Collectors.joining(","))));
		}
	}

	public boolean logDataForUser(String userLogin) {
		return DATA_IN_LOGS
				&& (USERS_WATCHED.isEmpty() || USERS_WATCHED.stream().anyMatch(u -> u.equalsIgnoreCase(userLogin)));
	}

	private void updateUsersListToWatch() throws IOException {
		if (!DATA_IN_LOGS) {
			USERS_WATCHED.clear();
			return;
		}

		Path path = Paths.get(ROOT, DATA_IN_LOG_FILENAME);
		List<String> lines = Files.readAllLines(path);
		lines.removeIf(l -> l.isBlank() || l.isEmpty());
		USERS_WATCHED.clear();
		if (lines.isEmpty()) {
			return;
		}
		USERS_WATCHED.addAll(lines);
		USERS_WATCHED = new HashSet<>(USERS_WATCHED);
		logger.debug("data.in.logs watch list [{}]", USERS_WATCHED.stream().collect(Collectors.joining(",")));
	}

	private void disablePolicies() {
		DISABLE_POLICIES = isFileExists(DISABLE_POLICIES_FILENAME);
	}

	private void dataInLogs() throws IOException {
		DATA_IN_LOGS = isFileExists(DATA_IN_LOG_FILENAME);
	}

	private void failOnInvalidRequests() {
		FAIL_ON_INVALID_REQUESTS = isFileExists(VALIDATE_REQUESTS_FILENAME);
	}

	public boolean changes() {
		boolean dataInLogsChanges = false;
		File file = getFile(DATA_IN_LOG_FILENAME);
		if (file.exists()) {
			long newSize = file.length();
			if (PREVIOUS_DATA_IN_LOGS_SIZE != newSize) {
				PREVIOUS_DATA_IN_LOGS_SIZE = newSize;
				dataInLogsChanges = true;
			}
		} else {
			if (PREVIOUS_DATA_IN_LOGS_SIZE > 0) {
				PREVIOUS_DATA_IN_LOGS_SIZE = -1;
				dataInLogsChanges = true;
			}
		}

		boolean invalidRequestsChanges = FAIL_ON_INVALID_REQUESTS && isFileExists(VALIDATE_REQUESTS_FILENAME);
		boolean disablePoliciesChanges = DISABLE_POLICIES && isFileExists(DISABLE_POLICIES_FILENAME);

		return dataInLogsChanges || invalidRequestsChanges || disablePoliciesChanges;
	}

	private static boolean isFileExists(String filename) {
		return getFile(filename).exists();
	}

	private static File getFile(String filename) {
		return new File(ROOT, filename);
	}

}
