/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2025
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.ui.adminconsole.directory.user;

import java.util.List;

import com.google.gwt.core.client.GWT;
import com.google.gwt.core.client.JavaScriptObject;
import com.google.gwt.dom.client.Style.Cursor;
import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.i18n.client.DateTimeFormat;
import com.google.gwt.i18n.client.DateTimeFormat.PredefinedFormat;
import com.google.gwt.i18n.client.NumberFormat;
import com.google.gwt.resources.client.ClientBundle;
import com.google.gwt.resources.client.CssResource;
import com.google.gwt.uibinder.client.UiBinder;
import com.google.gwt.uibinder.client.UiField;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.ui.Anchor;
import com.google.gwt.user.client.ui.CheckBox;
import com.google.gwt.user.client.ui.FlexTable;
import com.google.gwt.user.client.ui.FlowPanel;
import com.google.gwt.user.client.ui.HTMLTable.Cell;
import com.google.gwt.user.client.ui.Label;

import net.bluemind.core.api.ListResult;
import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.commons.gwt.JsMapStringJsObject;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.system.api.Credential;
import net.bluemind.system.api.gwt.endpoint.CredentialsGwtEndpoint;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.eas.api.Account;
import net.bluemind.eas.api.gwt.endpoint.EasGwtEndpoint;
import net.bluemind.gwtconsoleapp.base.editor.WidgetElement;
import net.bluemind.gwtconsoleapp.base.editor.gwt.CompositeGwtWidgetElement;
import net.bluemind.gwtconsoleapp.base.editor.gwt.GwtWidgetElement;
import net.bluemind.gwtconsoleapp.base.editor.gwt.IGwtDelegateFactory;
import net.bluemind.gwtconsoleapp.base.editor.gwt.IGwtWidgetElement;
import net.bluemind.gwtconsoleapp.base.handler.DefaultAsyncHandler;
import net.bluemind.gwtconsoleapp.base.notification.Notification;
import net.bluemind.ui.adminconsole.directory.user.l10n.UserConstants;
import net.bluemind.ui.common.client.forms.Ajax;
import net.bluemind.ui.common.client.icon.Trash;

import net.bluemind.system.api.SysConfKeys;
import net.bluemind.user.api.gwt.js.JsUser;


public class CredentialsPanel extends CompositeGwtWidgetElement {

	public static final String TYPE = "bm.ac.CredentialsEditor";

	public static void registerType() {
		GwtWidgetElement.register(TYPE, new IGwtDelegateFactory<IGwtWidgetElement, WidgetElement>() {

			@Override
			public IGwtWidgetElement create(WidgetElement e) {
				return new CredentialsPanel();
			}
		});
		GWT.log("bm.ac.CredentialsEditor registred");
	}

	private static CredentialsPanelUiBinder uiBinder = GWT.create(CredentialsPanelUiBinder.class);

	interface CredentialsPanelUiBinder extends UiBinder<FlowPanel, CredentialsPanel> {
	}

	public static interface Resources extends ClientBundle {

		@Source("CredentialsPanel.css")
		Style editStyle();

	}

	public static interface Style extends CssResource {

		String container();

		String header();

		String row();

		String icon();

		String refreshList();

	}

	@UiField
	FlexTable credentialsList;

	@UiField
	Anchor refreshList;

	private static final Resources res = GWT.create(Resources.class);

	private final Style s;
	private String domainUid;
	private String entryUid;

	protected CredentialsPanel() {
		s = res.editStyle();
		s.ensureInjected();
		initWidget(uiBinder.createAndBindUi(this));
		credentialsList.setStyleName(s.container());

		refreshList.setStyleName(s.refreshList());
		refreshList.getElement().setId("credentials-panel-refresh-list");
		refreshList.addClickHandler(new ClickHandler() {

			@Override
			public void onClick(ClickEvent event) {
				setDisplay(entryUid);
			}
		});
	}

	public void addCredentialRowHeader() {
		int idx = 0;
		credentialsList.setWidget(0, idx++, new Label(UserConstants.INST.identity()));
		credentialsList.setWidget(0, idx++, new Label(UserConstants.INST.type()));
		credentialsList.setWidget(0, idx++, new Label(UserConstants.INST.credentialLabel()));
		credentialsList.setWidget(0, idx++, new Label());
		credentialsList.getRowFormatter().setStyleName(0, s.header());
	}

	public void addCredentialRow(final Credential d) {
		int row = credentialsList.getRowCount();
		int idx = 0;

		credentialsList.getRowFormatter().setStylePrimaryName(row, s.row());

		credentialsList.setWidget(row, idx++, new Label(d.id));
		credentialsList.setWidget(row, idx++, new Label(d.type));
		credentialsList.setWidget(row, idx++, new Label(d.label));

		Trash trash = new Trash();
		trash.setId("credentials-panel-trash-" + d.id);
		trash.addStyleName(s.icon());
		trash.addClickHandler(new ClickHandler() {
			@Override
			public void onClick(ClickEvent event) {
				event.stopPropagation();
				Cell cell = credentialsList.getCellForEvent(event);
				int r = cell.getRowIndex();
				removeCredential(d, r);
			}
		});
		credentialsList.setWidget(row, idx++, trash);
	}

	private CredentialsGwtEndpoint getCredentialService() {
		return new CredentialsGwtEndpoint(Ajax.TOKEN.getSessionId(), this.domainUid);
	}

	private void removeCredential(Credential d, final int row) {
		if (Window.confirm(UserConstants.INST.confirmCredentialRemoval(d.id))) {
			getCredentialService().removeUserCredential(this.entryUid, d.id, new DefaultAsyncHandler<Void>() {
				@Override
				public void success(Void value) {
					credentialsList.removeRow(row);
					if (credentialsList.getRowCount() == 0) {
						setEmptyCredentialMessage();
					}
				}
			});
		}
	}

	private void setEmptyCredentialMessage() {
		credentialsList.setWidget(credentialsList.getRowCount(), 0, new Label(UserConstants.INST.noCredential()));
	}

	/**
	 * @param credentials
	 */
	private void setCredentials(List<Credential> credentials) {
		credentialsList.clear();
		credentialsList.removeAllRows();
		if (credentials.isEmpty()) {
			setEmptyCredentialMessage();
		} else {
			addCredentialRowHeader();
			credentials.stream().filter(d -> d != null).forEach(this::addCredentialRow);
		}
	}

	/**
	 * @param uid
	 */
	public void setDisplay(String uid) {
		this.entryUid = uid;
		credentialsList.clear();
		credentialsList.removeAllRows();

		getCredentialService().getObfuscatedUserCredentials(this.entryUid, new DefaultAsyncHandler<ListResult<Credential>>() {

			@Override
			public void success(ListResult<Credential> result) {
				setCredentials(result.values);
			}

			@Override
			public void failure(Throwable e) {
				if (e instanceof ServerFault && ((ServerFault) e).getCode() == ErrorCode.NOT_FOUND) {
					CredentialsPanel.this.asWidget().setVisible(false);
				} else {
					super.failure(e);
				}
			}
		});
	}

	@Override
	public void loadModel(JavaScriptObject model) {
		JsMapStringJsObject map = model.cast();
		entryUid = map.getString("userId");
		domainUid = map.getString("domainUid");
		setDisplay(entryUid);
	}
}
