package net.bluemind.core.backup.continuous.events.bodies;

import java.util.Optional;
import java.util.function.Supplier;

import com.github.benmanes.caffeine.cache.Cache;
import com.google.common.base.Suppliers;

import net.bluemind.backend.cyrus.partitions.CyrusPartition;
import net.bluemind.backend.mail.api.MessageBody;
import net.bluemind.backend.mail.replica.api.IDbMessageBodies;
import net.bluemind.backend.mail.replica.api.MailboxRecord;
import net.bluemind.backend.mail.replica.hook.IMessageBodyHook;
import net.bluemind.backend.mail.replica.hook.ISaveBodySupport;
import net.bluemind.core.backup.continuous.events.ContinuousContenairization;
import net.bluemind.core.caches.registry.CacheRegistry;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.directory.api.IDirectory;

public class MessageBodyHook
		implements IMessageBodyHook, ContinuousContenairization<MessageBody>, IBaseBodyHooks, ISaveBodySupport {

	@Override
	public String type() {
		return "message_bodies";
	}

	private static final Supplier<Cache<String, MessageBody>> cacheHolder = Suppliers
			.memoize(() -> CacheRegistry.get().get("net.bluemind.backend.mail.replica.service.internal.BodiesCache"));

	@Override
	public void preCreate(String domainUid, String ownerId, MailboxRecord mailboxRecord) {
		loadAndSave(domainUid, ownerId, mailboxRecord, true);
	}

	@Override
	public void preIndex(String guid, byte[] esSource) {
		// not now
	}

	private void loadAndSave(String domainUid, String ownerId, MailboxRecord mailboxRecord, boolean create) {
		if (targetOutbox(domainUid, ownerId).isPaused()) {
			return;
		}
		Optional.ofNullable(fetchMessageBody(domainUid, ownerId, mailboxRecord)).ifPresent(
				messageBody -> save(domainUid(), ownerId(messageBody.guid), messageBody.guid, messageBody, create));
	}

	public void save(MessageBody mb) {
		save(domainUid(), ownerId(mb.guid), mb.guid, mb, true);
	}

	@Override
	public void preUpdate(String domainUid, String ownerId, MailboxRecord mailboxRecord) {
		loadAndSave(domainUid, ownerId, mailboxRecord, false);
	}

	public static MessageBody fetchMessageBody(String domainUid, String ownerId, MailboxRecord mailboxRecord) {
		return Optional.ofNullable(cacheHolder.get().getIfPresent(mailboxRecord.messageBody))
				.orElseGet(() -> slowFetchMessageBody(domainUid, ownerId, mailboxRecord));
	}

	private static MessageBody slowFetchMessageBody(String domainUid, String ownerId, MailboxRecord mailboxRecord) {
		ServerSideServiceProvider prov = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM);
		IDirectory directoryApi = prov.instance(IDirectory.class, domainUid);
		var mailbox = directoryApi.findByEntryUid(ownerId);
		String partition = CyrusPartition.forServerAndDomain(mailbox.dataLocation, domainUid).name;

		IDbMessageBodies apiMessageBodies = prov.instance(IDbMessageBodies.class, partition);
		return apiMessageBodies.getComplete(mailboxRecord.messageBody);
	}

	@Override
	public void preDelete(String guid) {
		MessageBody mb = new MessageBody();
		mb.guid = guid;
		delete(domainUid(), ownerId(guid), guid, mb);
	}

}
