/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2021
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.core.backup.continuous.events;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.vertx.core.Vertx;
import io.vertx.core.eventbus.Message;
import io.vertx.core.json.JsonObject;
import net.bluemind.config.InstallationId;
import net.bluemind.config.Token;
import net.bluemind.core.backup.continuous.dto.CoreTok;
import net.bluemind.core.container.model.BaseContainerDescriptor;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.system.api.SystemState;
import net.bluemind.system.stateobserver.IStateListener;
import net.bluemind.user.api.User;
import net.bluemind.user.api.UsersHookAddress;
import net.bluemind.user.service.IInCoreUser;

public class TokenContinuousHook implements IStateListener {

	private static final Logger logger = LoggerFactory.getLogger(TokenContinuousHook.class);
	private SystemState cur;

	@Override
	public void stateChanged(SystemState newState) {
		if (newState == SystemState.CORE_STATE_RUNNING && newState != cur) {
			pushTokenAndPassword();
		}
		cur = newState;
	}

	@Override
	public void init(Vertx vx) {
		vx.eventBus().consumer(UsersHookAddress.PASSWORD_UPDATED, (Message<JsonObject> msg) -> {
			JsonObject js = msg.body();
			if (js.getString("domain").equals("global.virt")) {
				pushTokenAndPassword();
			}
		});
	}

	private void pushTokenAndPassword() {
		BaseContainerDescriptor instContainer = BaseContainerDescriptor.create(InstallationId.getIdentifier(),
				"inst name", "system", "installation", null, true);

		ServerSideServiceProvider prov = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM);
		IInCoreUser userApi = prov.instance(IInCoreUser.class, "global.virt");
		ItemValue<User> forUidLookup = userApi.byLogin("admin0");
		if (forUidLookup == null) {
			logger.error("admin0 account not found");
			return;
		}
		ItemValue<User> withPassword = userApi.getFull(forUidLookup.uid, false);
		if (withPassword == null) {
			logger.error("admin0 (getFull) account not found");
			return;
		}
		CoreTok token = CoreTok.of(Token.admin0(), withPassword.value.password);
		ItemValue<CoreTok> asItem = ItemValue.create("core_tok", token);
		asItem.internalId = -1;

		TxOutboxLookup.forContainer(instContainer).forKafka(asItem, null, false);
		logger.info("Pushed token {} to installation container", asItem);
	}

}
