/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.system.cql.store;

import java.io.File;
import java.util.List;

import com.datastax.oss.driver.api.core.CqlSession;
import com.typesafe.config.Config;
import com.typesafe.config.ConfigFactory;

import net.bluemind.core.container.model.ItemValue;
import net.bluemind.cql.CqlKeyspaceService;
import net.bluemind.cql.CqlSchemaService;
import net.bluemind.cql.CqlSessions;
import net.bluemind.cql.persistence.CqlKeyspaceStore;
import net.bluemind.cql.persistence.CqlSchemaStore;
import net.bluemind.network.topology.Topology;
import net.bluemind.server.api.Server;
import net.bluemind.server.api.TagDescriptor;
import net.bluemind.system.repository.IStorageSchemaStore;

public class CqlStorageSchemaStore implements IStorageSchemaStore {

	@Override
	public void initSchemas() {
		Config conf = ConfigFactory.parseFile(new File("/etc/bm/cql-persistence.conf"));
		List<String> nodes = List.of("127.0.0.1");
		if (conf.hasPath("persistence.scylla.nodes")) {
			nodes = conf.getStringList("persistence.scylla.nodes");
		}
		Server srv = new Server();
		srv.tags = List.of(TagDescriptor.cql_node.getTag());
		srv.ip = nodes.getFirst();
		ItemValue<Server> cassServerItem = ItemValue.create("scylla", srv);
		Topology.update(List.of(cassServerItem));

		try (CqlSession systemSession = CqlSessions.system()) {
			CqlKeyspaceStore store = new CqlKeyspaceStore(systemSession);
			CqlKeyspaceService service = new CqlKeyspaceService(store);
			List<String> spaces = service.initialize();
			for (String ks : spaces) {
				CqlSchemaStore ss = new CqlSchemaStore(CqlSessions.forKeyspace(ks));
				CqlSchemaService schemas = new CqlSchemaService(ks, ss);
				schemas.initialize();
			}
		}

	}

}
