import { openDB, DBSchema, IDBPDatabase } from "idb";
import logger from "@bluemind/logger";
import session from "@bluemind/session";

interface EnvironmentSchema extends DBSchema {
    system: {
        key: string;
        value: { key: string; value: string };
    };
}

const VERSION = 1;

interface EnvironmentDB {
    setMailboxCopyGuid(uid: string): Promise<void>;
    getMailboxCopyGuid(): Promise<string | undefined>;
    close(): Promise<void>;
}
export class EnvironmentDBImpl implements EnvironmentDB {
    db: Promise<IDBPDatabase<EnvironmentSchema>>;
    constructor(name: string) {
        this.db = this.openDB(name);
    }
    private async openDB(name: string) {
        return openDB<EnvironmentSchema>(`${name}:environment`, VERSION, {
            upgrade(db, oldVersion) {
                logger.log(`[SW][DB] Upgrading from ${oldVersion} to ${VERSION}`);
                if (oldVersion < VERSION) {
                    logger.log("[SW][DB] Upgrading deleting existing object store");
                    for (const name of Object.values(db.objectStoreNames)) {
                        db.deleteObjectStore(name);
                    }
                }
                db.createObjectStore("system", { keyPath: "key" });
            },
            blocking: async () => {
                (await this.db).close();
                this.db = this.openDB(name);
            }
        });
    }
    async close(): Promise<void> {
        (await this.db).close();
    }

    async setMailboxCopyGuid(uid: string) {
        logger.log(`[SW][DB] Initialize environment mailboxCopyGuid to ${uid}.`);
        await (await this.db).put("system", { key: "mailboxCopyGuid", value: uid });
    }

    async getMailboxCopyGuid() {
        const data = await (await this.db).get("system", "mailboxCopyGuid");
        if (data === undefined) {
            return undefined;
        }
        return data.value;
    }
}

let implementation: EnvironmentDB | null = null;
async function instance(): Promise<EnvironmentDB> {
    if (!implementation) {
        const name = `user.${await session.userId}@${(await session.domain).replace(/\./g, "_")}`;
        implementation = new EnvironmentDBImpl(name);
    }
    return implementation;
}

session.addEventListener("change", event => {
    const { old, value } = event.detail;
    if (value.userId != old?.userId && implementation) {
        implementation?.close();
        implementation = null;
    }
});

const db: EnvironmentDB = {
    setMailboxCopyGuid: uid => instance().then(db => db.setMailboxCopyGuid(uid)),
    getMailboxCopyGuid: () => instance().then(db => db.getMailboxCopyGuid()),
    close: () => instance().then(db => db.close())
};

export default db;
