/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.mailbox.backup;

import java.io.File;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import java.util.UUID;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.backend.mail.api.ISdsBackup;
import net.bluemind.config.Token;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.dataprotect.sdsspool.SdsDataProtectSpool;
import net.bluemind.network.topology.Topology;
import net.bluemind.sds.store.ISdsSyncStore;
import net.bluemind.sds.store.loader.SdsStoreLoader;
import net.bluemind.server.api.Server;
import net.bluemind.server.api.TagDescriptor;
import net.bluemind.serviceprovider.SPResolver;
import net.bluemind.system.api.ISystemConfiguration;
import net.bluemind.system.api.SystemConf;

public class MailSdsBackupServiceFactory implements ServerSideServiceProvider.IServerSideServiceFactory<ISdsBackup> {
	private static final Logger logger = LoggerFactory.getLogger(MailSdsBackupServiceFactory.class);

	@Override
	public Class<ISdsBackup> factoryClass() {
		return ISdsBackup.class;
	}

	@Override
	public ISdsBackup instance(BmContext context, String... params) throws ServerFault {
		Path tmpPath = Paths.get("/var/cache/bm-export/").resolve(UUID.randomUUID().toString()).toAbsolutePath();
		File dir = tmpPath.toFile();
		try {
			dir.mkdirs();
		} catch (SecurityException e) {
			throw new ServerFault("Cannot create file " + tmpPath + ": " + e.getMessage());
		}

		SystemConf config = SPResolver.get().resolve(Token.admin0()).instance(ISystemConfiguration.class).getValues();
		Map<String, ISdsSyncStore> sdsStores = new HashMap<>();
		SdsDataProtectSpool backupSpool = new SdsDataProtectSpool(tmpPath);
		for (ItemValue<Server> server : Topology.get().all(TagDescriptor.mail_imap.getTag())) {
			Optional<ISdsSyncStore> sdsSyncStore = new SdsStoreLoader().forSysconf(config, server.uid);
			if (sdsSyncStore.isEmpty()) {
				throw new ServerFault("no store at all for server= " + server.uid);
			}
			sdsStores.put(server.uid, sdsSyncStore.get());
			logger.info("loaded sds store for server={}: {}", server.uid, sdsSyncStore.get());
		}

		return new MailSdsBackupService(tmpPath, sdsStores, backupSpool);
	};

}