/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2021
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.mailbox.backup;

import java.io.IOException;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Sets;

import net.bluemind.backend.mail.api.ISdsBackup;
import net.bluemind.config.Token;
import net.bluemind.core.api.ListResult;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.IServiceProvider;
import net.bluemind.dataprotect.api.IDPContext;
import net.bluemind.dataprotect.api.PartGeneration;
import net.bluemind.dataprotect.common.backup.RepositoryBackupPath;
import net.bluemind.dataprotect.mailbox.deserializer.RestoreSdsMailbox;
import net.bluemind.dataprotect.sdsspool.SdsDataProtectSpool;
import net.bluemind.dataprotect.service.internal.CommonBackupWorker.MailboxIndexJsonList;
import net.bluemind.dataprotect.service.internal.ICommonBackupWorker;
import net.bluemind.dataprotect.service.internal.MailboxIndexJson;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.api.DirEntryQuery;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.domain.api.Domain;
import net.bluemind.domain.api.IDomains;
import net.bluemind.sds.store.ISdsSyncStore;
import net.bluemind.serviceprovider.SPResolver;
import net.bluemind.utils.FileUtils;

public class MailSdsBackupService implements ISdsBackup, ICommonBackupWorker {
	private static final Logger logger = LoggerFactory.getLogger(MailSdsBackupService.class);
	private final Path workingFolder;
	private final Path jsonIndex;
	private IDPContext ctx = new LogContext();

	private RestoreSdsMailbox lastBackup;
	IServiceProvider serviceProvider;

	private MailboxBackupWorker mailboxWorker;
	private MailshareBackupWorker mailshareWorker;
	private PartGeneration partGen;

	public MailSdsBackupService(Path workingFolder, PartGeneration partGen) {
		this.workingFolder = workingFolder;
		this.lastBackup = null;
		this.partGen = partGen;
		try {
			if (!FileUtils.isDirEmpty(this.workingFolder)) {
				this.lastBackup = new RestoreSdsMailbox(this.workingFolder);
			} else {
				logger.debug("Empty workingFolder: {}", this.workingFolder);
			}
		} catch (Exception e) {
			ctx.error(e, "Not able to read previous backup: {}", e.getMessage());
		}
		jsonIndex = this.workingFolder.resolve("index.json");
		serviceProvider = SPResolver.get().resolve(Token.admin0());

		mailboxWorker = new MailboxBackupWorker(ctx, workingFolder, jsonIndex, lastBackup, partGen);
		mailshareWorker = new MailshareBackupWorker(ctx, workingFolder, jsonIndex, lastBackup, partGen);

	}

	public MailSdsBackupService(Path lastBackupFolder, Map<String, ISdsSyncStore> productionStores,
			SdsDataProtectSpool backupStore) {
		this(lastBackupFolder, null);
		mailboxWorker.setProductionStores(productionStores);
		mailshareWorker.setProductionStores(productionStores);
		mailboxWorker.setBackupStore(backupStore);
		mailshareWorker.setBackupStore(backupStore);
	}

	public void setContext(IDPContext context) {
		if (context != null) {
			this.ctx = context;
		}
	}

	public Set<String> backupDomains(List<String> domainUids, boolean downloadEmailContent) {
		IDomains domainsService = serviceProvider.instance(IDomains.class);
		ctx.info(String.format("backup domains requested (%s)", domainUids.stream().collect(Collectors.joining(","))));

		List<MailboxIndexJson> mailboxRetList = new ArrayList<>();

		for (String domainUid : domainUids) {
			ItemValue<Domain> domain = domainsService.get(domainUid);
			IDirectory dirApi = serviceProvider.instance(IDirectory.class, domain.uid);
			ListResult<String> users = dirApi.searchUids(DirEntryQuery.filterKindWithHidden(Kind.USER));
			ListResult<String> mailshares = dirApi.searchUids(DirEntryQuery.filterKindWithHidden(Kind.MAILSHARE));
			ctx.info(String.format("backup domain %s: %d users, %d mailshares", domain.value.defaultAlias, users.total,
					mailshares.total));

			MailboxIndexJsonList mailboxJsonValues = mailboxWorker.runIt(domain, users.values, downloadEmailContent);
			MailboxIndexJsonList mailshareJsonValues = mailshareWorker.runIt(domain, mailshares.values,
					downloadEmailContent);

			mailboxRetList.addAll(mailboxJsonValues.mailboxesIndexInfo());
			mailboxRetList.addAll(mailshareJsonValues.mailboxesIndexInfo());
		}

		try {
			RepositoryBackupPath.writeIndexFileToDir(workingFolder, mailboxRetList);
		} catch (IOException e) {
			logError(e, "Error trying to write index.json {}", workingFolder);
		}

		return Sets.newHashSet(workingFolder.toString());
	}

	@Override
	public String backupMailbox(String domainUid, String dirEntryUid, boolean downloadEmailContent) throws ServerFault {
		manageBackupMailbox(domainUid, dirEntryUid, downloadEmailContent);
		return workingFolder.toAbsolutePath().toString();
	}

	private void manageBackupMailbox(String domainUid, String dirEntryUid, boolean downloadEmailContent) {
		ItemValue<Domain> domain = serviceProvider.instance(IDomains.class).get(domainUid);
		DirEntry de = serviceProvider.instance(IDirectory.class, domainUid).findByEntryUid(dirEntryUid);
		if (Kind.USER.equals(de.kind)) {
			mailboxWorker.backupSdsUser(de, domain, downloadEmailContent);
		} else if (Kind.MAILSHARE.equals(de.kind)) {
			mailshareWorker.backupSdsMailshare(de, domain, downloadEmailContent);
		} else if (Kind.DOMAIN.equals(de.kind)) {
			backupDomains(Arrays.asList(de.entryUid), downloadEmailContent);
		} else {
			throw new ServerFault("Don't know how to backup direntry " + de);
		}
	}

	@Override
	public void logError(Throwable throwable) {
		if (partGen != null) {
			partGen.withWarnings = true;
		}
		ctx.error(throwable, throwable.getMessage());
	}

	@Override
	public void logError(Exception ex, String message, Object... args) {
		if (partGen != null) {
			partGen.withWarnings = true;
		}
		ctx.error(ex, message, args);
	}

	@Override
	public void logError(String message, Object... args) {
		if (partGen != null) {
			partGen.withWarnings = true;
		}
		ctx.error("en", message, args);
	}

	private class LogContext implements IDPContext {

		public LogContext() {
		}

		@Override
		public ITool tool() {
			return null;
		}

		@Override
		public void info(String locale, String msg) {
			logger.info(msg);

		}

		@Override
		public void warn(String locale, String msg) {
			logger.warn(msg);

		}

		@Override
		public void error(String locale, String msg) {
			logger.error(msg);

		}

		@Override
		public void exception(Throwable ex) {
			StringWriter sw = new StringWriter();
			PrintWriter pw = new PrintWriter(sw);
			ex.printStackTrace(pw);
			logger.error(sw.toString());
		}

	}
}
