/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2025
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */

package net.bluemind.calendar.service.internal;

import java.sql.SQLException;
import java.util.Optional;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.calendar.api.internal.VEventHistory;
import net.bluemind.calendar.api.internal.VEventHistory.HistoryActionType;
import net.bluemind.calendar.repository.IVEventHistoryStore;
import net.bluemind.core.auditlogs.AuditLogEntry;
import net.bluemind.core.container.model.ChangeLogEntry.Type;
import net.bluemind.icalendar.api.ICalendarElement.Status;

public class VEventHistoryMapper extends CalendarAuditLogMapper {

	private static final Logger logger = LoggerFactory.getLogger(VEventHistoryMapper.class);

	private final IVEventHistoryStore veventHistoryStore;

	public VEventHistoryMapper(IVEventHistoryStore veventHistoryStore) {
		this.veventHistoryStore = veventHistoryStore;
	}

	public void store(AuditLogEntry entry) {
		if (entry.action.equals(Type.Deleted.name())) {
			return;
		}

		toVEventHistory(entry).ifPresent(hist -> {
			try {
				veventHistoryStore.store(hist);
			} catch (SQLException e) {
				logger.error("Error trying to store history on event", e);
			}
		});
	}

	private static Optional<VEventHistory> toVEventHistory(AuditLogEntry entry) {
		VEventHistory eventHistory = new VEventHistory();
		eventHistory.containerId = entry.container.internalId();
		eventHistory.itemId = entry.item.id();
		eventHistory.icsUid = entry.content.key();
		eventHistory.actionType = getActionType(entry);
		eventHistory.date = entry.timestamp;
		eventHistory.author = getAuthor(entry);
		eventHistory.origin = getOrigin(entry);

		boolean excludeSystemChanges = eventHistory.author.equals("sys")
				&& eventHistory.origin.equals("internal-system");
		return excludeSystemChanges ? Optional.empty() : Optional.of(eventHistory);
	}

	private static String getAuthor(AuditLogEntry entry) {
		if (entry.securityContext != null) {
			String value = entry.securityContext.email();
			if (value != null) {
				return value;
			}
			value = entry.securityContext.displayName();
			if (value != null) {
				return value;
			}
		}
		return entry.content.author() != null && !entry.content.author().isEmpty()
				? entry.content.author().stream().collect(Collectors.joining(", "))
				: "";
	}

	private static String getOrigin(AuditLogEntry entry) {
		return Optional.of(entry.securityContext).map((c -> entry.securityContext.origin())).orElse("");
	}

	private static HistoryActionType getActionType(AuditLogEntry entry) {
		HistoryActionType entryAction = HistoryActionType.valueOf(entry.action);
		return entryAction == HistoryActionType.Updated && Status.Cancelled.name().equals(entry.content.status())
				? HistoryActionType.Deleted
				: entryAction;
	}
}