/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.delivery.smtp.ndr;

import java.io.IOException;
import java.io.InputStream;
import java.util.Collection;
import java.util.List;
import java.util.Optional;

import org.apache.james.mime4j.codec.EncoderUtil;
import org.apache.james.mime4j.dom.address.Mailbox;
import org.apache.james.mime4j.field.address.AddressBuilder;
import org.apache.james.mime4j.field.address.ParseException;
import org.columba.ristretto.message.Address;
import org.columba.ristretto.smtp.SMTPException;
import org.columba.ristretto.smtp.SMTPProtocol;
import org.columba.ristretto.smtp.SMTPResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SendmailHelper {

	private static final Logger logger = LoggerFactory.getLogger(SendmailHelper.class);

	static final String SMTP_ERROR_STATUS = "5.0.0";
	static final String SMTP_OK_STATUS = "2.0.0";

	private SendmailHelper() {

	}

	public static Mailbox formatAddress(String displayName, String email) {
		try {
			return AddressBuilder.DEFAULT
					.parseMailbox((displayName != null ? EncoderUtil.encodeAddressDisplayName(displayName) : "") + " <"
							+ email.toLowerCase() + ">");
		} catch (ParseException e) {
			logger.error("addr parse exception for dn: {}, email:{}", displayName, email, e);
			return null;
		}
	}

	public static Optional<String> getSmtpErrorCode(int code, String message) {
		if (message == null || message.isEmpty()) {
			return Optional.of(isSuccessCode(code) ? SMTP_OK_STATUS : SMTP_ERROR_STATUS);
		}

		String status = message.split(" ")[0];
		// rfc3464: status-code = DIGIT "." 1*3DIGIT "." 1*3DIGIT
		if (status != null && !status.isEmpty() && status.matches("[2,4,5].\\d{1,3}.\\d{1,3}")) {
			return Optional.of(status);
		}
		return Optional.empty();
	}

	public static boolean isSuccessCode(int code) {
		return Integer.parseInt(Integer.toString(code).substring(0, 1)) == 2;
	}

	public static String getSmtpErrorCode(SMTPException sme) {
		return getSmtpErrorCode(sme.getCode(), sme.getMessage()).orElse(SMTP_ERROR_STATUS);
	}

	public static boolean isSenderUnauthorized(int code, String smtpStatus) {
		return code == 550 && smtpStatus.equals("5.7.1");
	}

	public static boolean isTooBigFile(int code, String smtpStatus) {
		return code == 552 && smtpStatus.equals("5.3.4");
	}

	public static boolean isNetworkError(int code) {
		return code >= 400 && code <= 500;
	}

	public static boolean isError(int code) {
		return code >= 400;
	}

	public static void smtpMailCmd(SMTPProtocol smtp, String fromAddress, Collection<Mailbox> recipients,
			List<FailedRecipient> failedRecipients) throws IOException, SMTPException {
		try {
			smtp.mail(new Address(fromAddress));
		} catch (SMTPException sme) {
			FailedRecipient.addErrorFailedRecipients(failedRecipients, sme,
					recipients.stream().map(Mailbox::getAddress).toList());
			throw sme;
		}
	}

	public static int smtpRcptCmd(SMTPProtocol smtp, Mailbox to, boolean requestedDSN, int requestedDSNs,
			List<FailedRecipient> failedRecipients) throws IOException {
		Address address = new Address(to.getAddress());
		try {
			if (requestedDSN) {
				requestedDSNs++;
				smtp.rcptWithDeliveryReport(address);
			} else {
				smtp.rcpt(address);
			}
			failedRecipients.add(FailedRecipient.ok(address.getMailAddress()));
		} catch (SMTPException sme) {
			failedRecipients.add(FailedRecipient.failure(sme, address.getMailAddress()));
		}
		return requestedDSNs;
	}

	public static SMTPResponse smtpDataCmd(SMTPProtocol smtp, InputStream is, Collection<Mailbox> recipients,
			List<FailedRecipient> failedRecipients) throws IOException, SMTPException {
		try {
			return smtp.data(is);
		} catch (SMTPException sme) {
			FailedRecipient.addErrorFailedRecipients(failedRecipients, sme,
					recipients.stream().map(Mailbox::getAddress).toList());
			throw sme;
		}
	}
}
