/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */

package net.bluemind.cli.adm.orphans.report;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.fusesource.jansi.Ansi;
import org.slf4j.helpers.MessageFormatter;

import net.bluemind.core.container.model.ItemValue;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.maintenance.checker.report.IReportLeaf;

public abstract class AbstractReport implements IReportLeaf {
	protected final String logId;
	protected final List<Log> logs = new ArrayList<>();
	protected final Set<String> repairs = new HashSet<>();
	protected final List<String> sqlCommands = new ArrayList<>();
	protected final Log.Level minLevel;
	private final Ansi ansi;

	protected AbstractReport(String logId, Log.Level minLevel) {
		this.logId = logId;
		this.minLevel = minLevel;
		ansi = Ansi.ansi();
	}

	public boolean hasErrors() {
		return logs.stream().anyMatch(m -> m.level.equals(Log.Level.ERROR));
	}

	public boolean hasWarning() {
		return logs.stream().anyMatch(m -> m.level.equals(Log.Level.WARNING));
	}

	public IReportLeaf info(String msg, Object... args) {
		return log(Log.Level.INFO, MessageFormatter.arrayFormat(msg, args).getMessage());
	}

	public IReportLeaf warn(String msg, Object... args) {
		return log(Log.Level.WARNING, MessageFormatter.arrayFormat(msg, args).getMessage());
	}

	public IReportLeaf error(String msg, Object... args) {
		return log(Log.Level.ERROR, MessageFormatter.arrayFormat(msg, args).getMessage());
	}

	private IReportLeaf log(Log.Level level, String msg, Object... args) {
		logs.add(new Log(level, MessageFormatter.arrayFormat(msg, args).getMessage()));
		return this;
	}

	public IReportLeaf repair(ItemValue<DirEntry> de, String repair, boolean autoRepair) {
		repairs.add(repair);
		return this;
	}

	public IReportLeaf repair(Runnable op) {
		op.run();
		return this;
	}

	public String toString() {
		List<String> msgs = new ArrayList<>();
		msgs.add(logId + ": " + (hasErrors() ? ansi.fgRed().a("ERROR").reset() : ansi.fgGreen().a("OK").reset()));
		logs.stream().filter(l -> l.level.ordinal() >= minLevel.ordinal())
				.forEach(s -> msgs.add("    [" + s.level + "]: " + s.message));
		return msgs.stream().collect(Collectors.joining("\n"));
	}

	public Log.Level level() {
		return minLevel;
	}
}
