/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2023
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.system.service.internal;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.vertx.core.AbstractVerticle;
import io.vertx.core.Verticle;
import io.vertx.core.eventbus.Message;
import io.vertx.core.json.JsonObject;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.IServiceProvider;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.domain.api.Domain;
import net.bluemind.domain.api.IDomainSettings;
import net.bluemind.domain.api.IInCoreDomains;
import net.bluemind.domain.hook.DomainHookAdapter;
import net.bluemind.hornetq.client.MQ;
import net.bluemind.hornetq.client.Shared;
import net.bluemind.lib.vertx.IUniqueVerticleFactory;
import net.bluemind.lib.vertx.IVerticleFactory;

public class SharedDomainSettingsVerticle extends AbstractVerticle {

	public static class Init implements IVerticleFactory, IUniqueVerticleFactory {
		@Override
		public boolean isWorker() {
			return true;
		}

		@Override
		public Verticle newInstance() {
			return new SharedDomainSettingsVerticle();
		}
	}

	public static class SharedDomainSettingsDomainHook extends DomainHookAdapter {
		@Override
		public void onUpdated(BmContext context, ItemValue<Domain> previousValue, ItemValue<Domain> domain)
				throws ServerFault {
			new SharedDomainSettingsVerticle().putDomainSettingsAndProperties(domain, Optional.empty());
		}

		@Override
		public void onPropertiesUpdated(BmContext context, ItemValue<Domain> domain,
				Map<String, String> previousProperties, Map<String, String> currentProperties) throws ServerFault {
			new SharedDomainSettingsVerticle().putDomainSettingsAndProperties(domain, Optional.empty());
		}

		@Override
		public void onSettingsUpdated(BmContext context, ItemValue<Domain> domain, Map<String, String> previousSettings,
				Map<String, String> currentSettings) throws ServerFault {
			new SharedDomainSettingsVerticle().putDomainSettingsAndProperties(domain, Optional.of(currentSettings));
		}

		@Override
		public void onDeleted(BmContext context, ItemValue<Domain> domain) throws ServerFault {
			new SharedDomainSettingsVerticle().deleteDomainSettings(domain);
		}
	}

	private static final Logger logger = LoggerFactory.getLogger(SharedDomainSettingsVerticle.class);

	@Override
	public void start() {
		vertx.eventBus().consumer("redis.storage.reset", this::updateOnStorageReset);
		updateAllDomainSettings();
	}

	private void updateAllDomainSettings() {
		MQ.init().thenAccept(v -> {
			IServiceProvider sysprov = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM);
			sysprov.instance(IInCoreDomains.class).allUnfiltered().stream().forEach(dom -> {
				putDomainSettingsAndProperties(dom, Optional.empty());
				logger.info("SharedDomainPropertiesVerticle pre-load domain properties for {}", dom.uid);
			});
		}).exceptionally(t -> {
			logger.warn(t.getMessage());
			return null;
		});
	}

	private void updateOnStorageReset(Message<JsonObject> event) {
		updateAllDomainSettings();
	}

	private void putDomainSettingsAndProperties(ItemValue<Domain> domain,
			Optional<Map<String, String>> domainSettings) {
		Map<String, String> infos = new HashMap<>();
		infos.putAll(domain.value.properties);
		infos.putAll(domainSettings.orElseGet(() -> ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM)
				.instance(IDomainSettings.class, domain.uid).get()));

		Shared.mapDomainSettings().put(domain.uid, infos);
	}

	private void deleteDomainSettings(ItemValue<Domain> domain) {
		Shared.mapDomainSettings().remove(domain.uid);
	}
}
