/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.service.tool;

import java.io.IOException;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import net.bluemind.common.io.Buffered;
import net.bluemind.core.api.fault.ServerFault;

public class ZipBuilder {
	public final String zipFileName;
	private Path tmpZip;
	private OutputStream baos;
	private ZipOutputStream zos;

	public static final Path WORKING_PATH = buildWorkingPath();

	static Path buildWorkingPath() {
		Path w = Paths.get("/var/spool/cyrus/data/dataprotect");
		try {
			Files.createDirectories(w);
		} catch (IOException e) {
			throw new ServerFault(e);
		}
		return w;
	}

	public ZipBuilder(String zipFileName) {
		this.zipFileName = zipFileName;
		try {
			this.tmpZip = Files.createTempFile(WORKING_PATH, "zip-builder", ".zip");
			this.baos = Buffered.output(Files.newOutputStream(tmpZip));
			this.zos = new ZipOutputStream(baos);
		} catch (IOException e) {
			throw new ServerFault(e);
		}
	}

	public void addFileEntry(String entryName, Path filePath) throws IOException {
		ZipEntry entry = new ZipEntry(entryName);
		zos.putNextEntry(entry);
		Files.copy(filePath, zos);
		zos.closeEntry();
	}

	public void writeToFile(Path outputPath) throws IOException {
		zos.finish();
		zos.close();
		baos.close();
		Files.move(tmpZip, outputPath, StandardCopyOption.REPLACE_EXISTING);
	}

	public void addDirectoryEntry(String directoryName) throws IOException {
		if (!directoryName.endsWith("/")) {
			directoryName += "/";
		}
		ZipEntry entry = new ZipEntry(directoryName);
		zos.putNextEntry(entry);
		zos.closeEntry();
	}

}
