/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.addressbook.service.internal;

import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.addressbook.api.CertInfo;
import net.bluemind.addressbook.api.IAddressBook;
import net.bluemind.addressbook.api.IAddressBookUids;
import net.bluemind.addressbook.api.IAddressBooks;
import net.bluemind.addressbook.api.VCard;
import net.bluemind.addressbook.api.VCardInfo;
import net.bluemind.addressbook.api.VCardQuery;
import net.bluemind.addressbook.api.VCardQuery.OrderBy;
import net.bluemind.addressbook.service.IInCoreAddressBook;
import net.bluemind.core.api.ListResult;
import net.bluemind.core.api.fault.ErrorCode;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.api.ContainerSubscriptionDescriptor;
import net.bluemind.core.container.model.ItemContainerValue;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.user.api.IUserSubscription;
import net.bluemind.utils.CertificateUtils;

public class AddressBooksService implements IAddressBooks {

	private static final Logger logger = LoggerFactory.getLogger(AddressBooksService.class);

	private SecurityContext securityContext;
	private BmContext context;

	public AddressBooksService(BmContext context) {
		this.context = context;
		this.securityContext = context.getSecurityContext();
	}

	@Override
	public ListResult<ItemContainerValue<VCardInfo>> search(VCardQuery query) throws ServerFault {
		if (securityContext.isAnonymous()) {
			throw new ServerFault("unauthenticated request", ErrorCode.PERMISSION_DENIED);
		}
		List<ContainerSubscriptionDescriptor> containers = context.provider()
				.instance(IUserSubscription.class, context.getSecurityContext().getContainerUid())
				.listSubscriptions(securityContext.getSubject(), IAddressBookUids.TYPE);

		// make container order pertinant too
		if (query.orderBy == OrderBy.Pertinance) {
			Collections.sort(containers, (a, b) -> {
				// domain addressbook first
				int ret = -Boolean.compare(a.owner.equals(securityContext.getContainerUid()),
						b.owner.equals(securityContext.getContainerUid()));
				if (ret != 0) {
					return ret;
				}

				// then user container
				ret = -Boolean.compare(a.owner.equals(securityContext.getSubject()),
						b.owner.equals(securityContext.getSubject()));
				if (ret != 0) {
					return ret;
				}

				// finally default container
				return -Boolean.compare(a.defaultContainer, b.defaultContainer);
			});
		}
		List<ItemContainerValue<VCardInfo>> res = new LinkedList<>();
		int total = 0;
		for (ContainerSubscriptionDescriptor container : containers) {

			try {
				IAddressBook service = context.provider().instance(IAddressBook.class, container.containerUid);

				ListResult<ItemValue<VCardInfo>> vcards = service.search(query);
				for (ItemValue<VCardInfo> vcardInfo : vcards.values) {
					if (vcardInfo != null) {
						ItemContainerValue<VCardInfo> v = adaptAsItemContainerValue(container.containerUid, vcardInfo);
						res.add(v);
					}
				}
				total += vcards.total;
			} catch (ServerFault e) {
				if (e.getCode() == ErrorCode.PERMISSION_DENIED) {
					logger.warn("user {} try to access {} but doesnt have right",
							securityContext.getSubject() + "@" + securityContext.getContainerUid(),
							container.containerUid);
				} else {
					throw e;
				}
			}
			if (query.size > 0 && res.size() >= query.size) {
				break;
			}
		}

		if (query.orderBy != OrderBy.Pertinance) {
			Collections.sort(res, (o1, o2) -> o1.displayName.compareTo(o2.displayName));
		}
		ListResult<ItemContainerValue<VCardInfo>> ret = new ListResult<>();
		ret.values = res;
		ret.total = total;
		return ret;
	}

	private <T> ItemContainerValue<T> adaptAsItemContainerValue(String containerUid, ItemValue<T> item) {
		ItemContainerValue<T> ret = new ItemContainerValue<>();
		ret.created = item.created;
		ret.updated = item.updated;
		ret.createdBy = item.createdBy;
		ret.updatedBy = item.updatedBy;
		ret.uid = item.uid;
		ret.version = item.version;
		ret.externalId = item.externalId;
		ret.displayName = item.displayName;
		ret.value = item.value;
		ret.containerUid = containerUid;
		return ret;
	}

	@Override
	public List<String> findUidsByEmail(String email) throws ServerFault {
		List<ContainerSubscriptionDescriptor> containers = context.provider()
				.instance(IUserSubscription.class, securityContext.getContainerUid())
				.listSubscriptions(securityContext.getSubject(), IAddressBookUids.TYPE);
		List<String> vcardUids = new ArrayList<>();
		for (ContainerSubscriptionDescriptor container : containers) {
			IInCoreAddressBook service = context.provider().instance(IInCoreAddressBook.class, container.containerUid);
			Optional.ofNullable(service.findByEmail(email)).ifPresent(vcardUids::addAll);
		}
		return vcardUids;
	}

	@Override
	public List<CertInfo> findCertsByEmail(String email) throws ServerFault {
		List<ContainerSubscriptionDescriptor> containers = context.provider()
				.instance(IUserSubscription.class, securityContext.getContainerUid())
				.listSubscriptions(securityContext.getSubject(), IAddressBookUids.TYPE);
		List<VCard.Security.Key> vcardUids = new ArrayList<>();
		for (ContainerSubscriptionDescriptor container : containers) {
			IInCoreAddressBook service = context.provider().instance(IInCoreAddressBook.class, container.containerUid);
			Optional.ofNullable(service.findByEmail(email)).map(idStrs -> idStrs.stream().map(Long::parseLong).toList())
					.map(uids -> service.multipleGetById(uids).stream()//
							.filter(vcf -> vcf.value.security != null && vcf.value.security.keys != null)
							.flatMap(vcf -> vcf.value.security.keys.stream().filter(k -> k.value != null)).toList())
					.ifPresent(vcardUids::addAll);
		}

		return vcardUids.stream().map(key -> {
			try {
				byte[] rawCertBytes = getCertificateFromPEM(key.value);
				X509Certificate x509 = CertificateUtils.generateX509Certificate(rawCertBytes);

				boolean[] usage = x509.getKeyUsage();
				CertInfo ci = new CertInfo();
				ci.usage = Arrays.stream(CertInfo.CertUsage.values()).filter(cu -> usage[cu.pos()])
						.collect(Collectors.toSet());
				ci.x509Certificate = rawCertBytes;
				return ci;
			} catch (Exception e) {
				logger.warn("Cert usage extraction error: {}", e.getMessage(), e);
				return null;
			}
		}).filter(Objects::nonNull).toList();
	}

	private static byte[] getCertificateFromPEM(String pemCertificate) {
		String cleanedCert = pemCertificate.replaceAll("\\s", "").replaceAll("-----[^-]+-----", "");
		return Base64.getDecoder().decode(cleanedCert);
	}

}
