/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.keycloak.utils;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import io.vertx.core.http.HttpMethod;
import io.vertx.core.json.JsonArray;
import io.vertx.core.json.JsonObject;
import net.bluemind.keycloak.api.IKeycloakUids;
import net.bluemind.keycloak.utils.endpoint.KeycloakEndpoints;

public class BlueMindFlowManager {
	private static final Logger logger = LoggerFactory.getLogger(BlueMindFlowManager.class);

	private static final String BM_AUTHENTICATOR_PROVIDER_ID = "bm-auth-usr-pwd-pubpriv-form";
	private static final String USER_SESSION_LIMITS = "user-session-limits";

	private final String realmId;

	private JsonArray domainFlowExecutions;

	public static BlueMindFlowManager fromCopy(String domainUid) {
		JsonObject reqData = new JsonObject();
		reqData.put("newName", IKeycloakUids.BLUEMIND_FLOW_ALIAS);
		KeycloakAdminClient.getInstance().call(
				KeycloakEndpoints.copyFlowEndpoint(IKeycloakUids.realmId(domainUid), IKeycloakUids.KEYCLOAK_FLOW_ALIAS),
				HttpMethod.POST, reqData);

		return forDomain(domainUid);
	}

	public static BlueMindFlowManager forDomain(String domainUid) {
		return new BlueMindFlowManager(domainUid).refreshDomainFlowExecutions().deleteDefaults()
				.refreshDomainFlowExecutions();
	}

	private BlueMindFlowManager(String domainUid) {
		this.realmId = IKeycloakUids.realmId(domainUid);
	}

	private BlueMindFlowManager refreshDomainFlowExecutions() {
		domainFlowExecutions = KeycloakAdminClient.getInstance()
				.call(KeycloakEndpoints.flowExecutionsEndpoint(realmId, IKeycloakUids.BLUEMIND_FLOW_ALIAS),
						HttpMethod.GET)
				.getJsonArray("results");
		return this;
	}

	private BlueMindFlowManager deleteDefaults() {
		List<JsonObject> defaultFormFlowExecutions = getFormExecutionsOf("auth-username-password-form");
		if (defaultFormFlowExecutions.isEmpty()) {
			return this;
		}

		// Remove default form flow execution
		defaultFormFlowExecutions.stream().map(defaultFormFlowExecution -> defaultFormFlowExecution.getString("id"))
				.filter(Objects::nonNull)
				.forEach(defaultFormFlowExecutionId -> KeycloakAdminClient.getInstance().call(
						KeycloakEndpoints.authenticationExecutionEndpoint(realmId, defaultFormFlowExecutionId),
						HttpMethod.DELETE));

		String organisationFlowId = getOrganizationFlowId();
		if (organisationFlowId != null) {
			KeycloakAdminClient.getInstance().call(
					KeycloakEndpoints.authenticationExecutionEndpoint(realmId, organisationFlowId), HttpMethod.DELETE);
		}

		return this;
	}

	private String getOrganizationFlowId() {
		for (int i = 0; i < domainFlowExecutions.size(); i++) {
			JsonObject currentDomainFlowExecution = domainFlowExecutions.getJsonObject(i);
			if (currentDomainFlowExecution.getInteger("level") == 0
					&& Boolean.TRUE.equals(currentDomainFlowExecution.getBoolean("authenticationFlow", Boolean.FALSE))
					&& currentDomainFlowExecution.getString("displayName").endsWith("Organization"))
				return currentDomainFlowExecution.getString("id");
		}

		return null;
	}

	private String getFormsFlowFlowId() {
		for (int i = 0; i < domainFlowExecutions.size(); i++) {
			JsonObject currentDomainFlowExecution = domainFlowExecutions.getJsonObject(i);
			if (currentDomainFlowExecution.getInteger("level") == 0
					&& Boolean.TRUE.equals(currentDomainFlowExecution.getBoolean("authenticationFlow", Boolean.FALSE))
					&& currentDomainFlowExecution.getString("displayName").endsWith("forms"))
				return currentDomainFlowExecution.getString("flowId");
		}

		return null;
	}

	private List<JsonObject> getFormExecutionsOf(String providerId) {
		List<JsonObject> formExecutions = new ArrayList<>();
		for (int i = 0; i < domainFlowExecutions.size(); i++) {
			JsonObject currentDomainFlowExecution = domainFlowExecutions.getJsonObject(i);
			if (providerId.equals(currentDomainFlowExecution.getString("providerId"))) {
				formExecutions.add(currentDomainFlowExecution);
			}
		}

		return formExecutions;
	}

	public BlueMindFlowManager setupBluemindAuthenticator() {
		List<JsonObject> bmFormExecutions = getFormExecutionsOf(BM_AUTHENTICATOR_PROVIDER_ID);
		if (bmFormExecutions.size() > 1) {
			bmFormExecutions.stream().map(bmFormExecution -> bmFormExecution.getString("id")).filter(Objects::nonNull)
					.forEach(bmFormExecutionId -> KeycloakAdminClient.getInstance().call(
							KeycloakEndpoints.authenticationExecutionEndpoint(realmId, bmFormExecutionId),
							HttpMethod.DELETE));

			refreshDomainFlowExecutions();
			bmFormExecutions = getFormExecutionsOf(BM_AUTHENTICATOR_PROVIDER_ID);
		}

		if (bmFormExecutions.isEmpty()) {
			JsonObject bmFormExec = new JsonObject();
			bmFormExec.put("authenticator", BM_AUTHENTICATOR_PROVIDER_ID);
			bmFormExec.put("authenticatorFlow", false);
			bmFormExec.put("requirement", "REQUIRED");
			bmFormExec.put("priority", 10);
			bmFormExec.put("parentFlow", getFormsFlowFlowId());
			KeycloakAdminClient.getInstance().call(KeycloakEndpoints.authenticationExecutionsEndpoint(realmId),
					HttpMethod.POST, bmFormExec);

			// Reload from Keycloak
			refreshDomainFlowExecutions();
			bmFormExecutions = getFormExecutionsOf(BM_AUTHENTICATOR_PROVIDER_ID);

			if (bmFormExecutions.size() != 1) {
				logger.error("Unable to setup " + BM_AUTHENTICATOR_PROVIDER_ID);
				return this;
			}

			String bmFormExecutionId = bmFormExecutions.get(0).getString("id");
			if (bmFormExecutionId != null) {
				KeycloakAdminClient.getInstance().call(
						KeycloakEndpoints.authenticationExecutionRaisePriorityEndpoint(realmId, bmFormExecutionId),
						HttpMethod.POST, null);
			}
		}

		return this;
	}

	public BlueMindFlowManager setupSessionLimits() {
		List<JsonObject> sessionLimits = getFormExecutionsOf(USER_SESSION_LIMITS);
		if (sessionLimits.size() > 1) {
			sessionLimits.stream().map(sessionLimit -> sessionLimit.getString("id")).filter(Objects::nonNull)
					.forEach(sessionLimitId -> KeycloakAdminClient.getInstance().call(
							KeycloakEndpoints.authenticationExecutionEndpoint(realmId, sessionLimitId),
							HttpMethod.DELETE));

			refreshDomainFlowExecutions();
			sessionLimits = getFormExecutionsOf(USER_SESSION_LIMITS);
		}

		if (sessionLimits.isEmpty()) {
			JsonObject bmSessionLimits = new JsonObject();
			bmSessionLimits.put("authenticator", USER_SESSION_LIMITS);
			bmSessionLimits.put("authenticatorFlow", false);
			bmSessionLimits.put("requirement", "REQUIRED");
			bmSessionLimits.put("parentFlow", getFormsFlowFlowId());
			KeycloakAdminClient.getInstance().call(KeycloakEndpoints.authenticationExecutionsEndpoint(realmId),
					HttpMethod.POST, bmSessionLimits);

			// Reload from Keycloak
			refreshDomainFlowExecutions();
			sessionLimits = getFormExecutionsOf(USER_SESSION_LIMITS);

			if (sessionLimits.size() != 1) {
				logger.error("Unable to setup " + USER_SESSION_LIMITS);
				return this;
			}

			JsonObject config = new JsonObject();
			config.put("userRealmLimit", 5);
			config.put("userClientLimit", 0);
			config.put("behavior", "Terminate oldest session");
			config.put("errorMessage", "");

			JsonObject parameters = new JsonObject();
			parameters.put("alias", "bm-session-limits");
			parameters.put("config", config);

			String sessionLimitId = sessionLimits.get(0).getString("id");
			if (sessionLimitId != null) {
				KeycloakAdminClient.getInstance().call(
						KeycloakEndpoints.authenticationExecutionConfigEndpoint(realmId, sessionLimitId),
						HttpMethod.POST, parameters);
			}
		}

		return this;
	}
}
