import { computed, markRaw } from "vue";

import { AlertTypes, NEW_LINE } from "@bluemind/bluetiful";
import { MimeType, InlineImageHelper } from "@bluemind/email";
import { preventStyleInvading, removeDuplicatedIds, sanitizeHtml } from "@bluemind/html-utils";
import { messageUtils, draftUtils, partUtils } from "@bluemind/mail";
import store from "@bluemind/store";

import { FETCH_PART_DATA } from "~/actions";
import ReplyToAlert from "~/components/MailAlerts/ReplyToAlert.vue";
import {
    SET_MESSAGE_HEADERS,
    SET_MESSAGE_SUBJECT,
    SET_MESSAGE_TO,
    SET_MESSAGE_CC,
    SET_MESSAGE_STRUCTURE,
    SET_DRAFT_EDITOR_CONTENT
} from "~/mutations";

import { getIdentityForReplyOrForward, setFrom } from "../ComposerFrom";

import { buildForwardStructure, buildReplyStructure } from "./initStructure";
const { MessageCreationModes, MessageHeader } = messageUtils;
const {
    getEditorContent,
    computeToRecipients,
    handleIdentificationFields,
    computeCcRecipients,
    quotePreviousMessage,
    computeSubject,
    COMPOSER_CAPABILITIES
} = draftUtils;

const { getPartsFromCapabilities } = partUtils;

export default async function initReplyOrForward(message, creationMode, previous, userPrefTextOnly, dispatchAlert) {
    const {
        message: previousMessage,
        attachments: previousAttachments,
        inlinePartsByCapabilities: previousInlinePartsByCapabilities
    } = previous;
    const previousInlines = getPartsFromCapabilities(
        { inlinePartsByCapabilities: previousInlinePartsByCapabilities },
        COMPOSER_CAPABILITIES
    );
    const identity = getIdentityForReplyOrForward(previousMessage);

    const identificationHeaders = handleIdentificationFields(previousMessage);
    const draftHeader = createDraftHeader(creationMode, previousMessage);
    if (previousMessage.replyTo?.length && previousMessage.from.address !== previousMessage.replyTo[0].address) {
        setAlertReplyTo(previousMessage.from, previousMessage.replyTo[0], creationMode, dispatchAlert);
    }
    store.commit(`mail/${SET_MESSAGE_HEADERS}`, {
        messageKey: message.key,
        headers: [draftHeader, ...identificationHeaders]
    });
    const subject = computeSubject(creationMode, previousMessage);
    store.commit(`mail/${SET_MESSAGE_SUBJECT}`, { messageKey: message.key, subject: subject });

    if (creationMode === MessageCreationModes.REPLY_ALL || creationMode === MessageCreationModes.REPLY) {
        const to = computeToRecipients(creationMode, previousMessage, identity);
        store.commit(`mail/${SET_MESSAGE_TO}`, { messageKey: message.key, to });
        const cc = computeCcRecipients(creationMode, previousMessage, identity)?.filter(
            contact => !message.to.some(to => to.address === contact.address)
        );
        store.commit(`mail/${SET_MESSAGE_CC}`, { messageKey: message.key, cc });
    }

    await createEditorContent(creationMode, userPrefTextOnly, previousMessage, previousInlines);
    await setFrom(identity, message);
    const structure = buildMessageStructure(creationMode, previousInlines, previousAttachments);
    store.commit(`mail/${SET_MESSAGE_STRUCTURE}`, { messageKey: message.key, structure });

    return message;
}
function setAlertReplyTo(from, replyTo, creationMode, dispatchAlert) {
    const payload = { from };
    if (creationMode === MessageCreationModes.REPLY) {
        payload.replyTo = replyTo;
    }
    if (creationMode === MessageCreationModes.REPLY || creationMode === MessageCreationModes.REPLY_ALL) {
        const alert = {
            uid: "REPLY_TO_ALERT",
            renderer: {
                component: markRaw(ReplyToAlert),
                props: {
                    payload
                }
            },
            type: AlertTypes.INFO
        };
        dispatchAlert(alert);
    }
}

function createDraftHeader(creationMode, { remoteRef, folderRef }) {
    const draftInfoHeader = {
        type: creationMode,
        messageInternalId: remoteRef.internalId,
        folderUid: folderRef.uid
    };
    return { name: MessageHeader.X_BM_DRAFT_INFO, values: [JSON.stringify(draftInfoHeader)] };
}

function buildMessageStructure(creationMode, inlines, attachments) {
    return creationMode === MessageCreationModes.FORWARD
        ? buildForwardStructure(inlines, attachments)
        : buildReplyStructure(inlines);
}

async function createEditorContent(creationMode, userPrefTextOnly, previousMessage, inlines) {
    await store.dispatch(`mail/${FETCH_PART_DATA}`, {
        messageKey: previousMessage.key,
        folderUid: previousMessage.folderRef.uid,
        imapUid: previousMessage.remoteRef.imapUid,
        parts: inlines.filter(
            part => MimeType.isHtml(part) || MimeType.isText(part) || (MimeType.isImage(part) && part.contentId)
        )
    });
    const partsByMessageKey = computed(() => store.state.mail.partsData.partsByMessageKey);

    let contentFromPreviousMessage = getEditorContent(
        userPrefTextOnly,
        inlines,
        partsByMessageKey.value[previousMessage.key],
        store.state.settings.lang
    );

    if (!userPrefTextOnly) {
        const partsWithCid = inlines.filter(part => MimeType.isImage(part) && part.contentId);
        const insertionResult = await InlineImageHelper.insertAsBase64(
            [contentFromPreviousMessage],
            partsWithCid,
            partsByMessageKey.value[previousMessage.key]
        );
        contentFromPreviousMessage = insertionResult.contentsWithImageInserted[0];
        contentFromPreviousMessage = sanitizeHtml(contentFromPreviousMessage);
        contentFromPreviousMessage = preventStyleInvading(contentFromPreviousMessage);
        contentFromPreviousMessage = removeDuplicatedIds(contentFromPreviousMessage);
    }
    const quote = quotePreviousMessage(contentFromPreviousMessage, previousMessage, creationMode, userPrefTextOnly);
    store.commit(`mail/${SET_DRAFT_EDITOR_CONTENT}`, NEW_LINE + quote + NEW_LINE);
}
