/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2018
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.cli.user;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Arrays;
import java.util.Optional;
import java.util.UUID;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.backend.mail.api.ISdsBackup;
import net.bluemind.cli.calendar.ExportCalendarCommand;
import net.bluemind.cli.cmd.api.CliException;
import net.bluemind.cli.cmd.api.ICmdLet;
import net.bluemind.cli.cmd.api.ICmdLetRegistration;
import net.bluemind.cli.contact.ExportAddressBookCommand;
import net.bluemind.cli.directory.common.SingleOrDomainOperation;
import net.bluemind.cli.notes.ExportNotesCommand;
import net.bluemind.cli.todolist.ExportTodolistCommand;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.domain.api.Domain;
import net.bluemind.domain.api.IDomains;
import net.bluemind.utils.FileUtils;
import picocli.CommandLine.Command;
import picocli.CommandLine.Option;

@Command(name = "export", description = "export user data to an archive file")
public class UserExportCommand extends SingleOrDomainOperation {
	private static final int BUFFER_SIZE = 8192;

	private static final Logger logger = LoggerFactory.getLogger(UserExportCommand.class);

	public static class Reg implements ICmdLetRegistration {

		@Override
		public Optional<String> group() {
			return Optional.of("user");
		}

		@Override
		public Class<? extends ICmdLet> commandClass() {
			return UserExportCommand.class;
		}
	}

	private Path defaultPath = Paths.get("/var/cache/bm-export/");

	@Option(names = { "-o",
			"--output-directory" }, defaultValue = "/var/cache/bm-export/", description = "output directory of exported users")
	public Path exportDirectoryPath = defaultPath;

	@Option(names = "--email-content", description = "download email messages content", negatable = true, defaultValue = "true", fallbackValue = "true")
	public boolean downloadEmailContent;

	@Override
	public void synchronousDirOperation(String domainUid, ItemValue<DirEntry> de) {
		// BM-15290: Needed when using --match [a-c].*
		Path outputDir = defaultPath.resolve(UUID.randomUUID().toString()).toAbsolutePath();
		ItemValue<Domain> domain = ctx.adminApi().instance(IDomains.class).get(domainUid);
		if (domain == null) {
			ctx.error("Unable to retrieve domain uid {}", domainUid);
		}
		File dir = outputDir.toFile();
		File exportDir = exportDirectoryPath.toFile();
		Path mailPath = null;
		try {
			dir.mkdirs();
			exportDir.mkdirs();
			Arrays.asList("contact", "calendar", "task", "notes")
					.forEach(data -> exportData(outputDir, domain, de, data));
			mailPath = exportMailData(domain, de);
			ctx.info("Creating archive file, can take a moment...");
			File archiveFile = createArchive(outputDir, mailPath, de);
			ctx.info("Archive file for " + de.value.email + " created as : " + archiveFile.getAbsolutePath());
		} finally {
			FileUtils.delete(dir);
			if (mailPath != null) {
				FileUtils.delete(mailPath.toFile());
			}
		}

	}

	private File createArchive(Path outputDir, Path emailOutputDir, ItemValue<DirEntry> de) {
		Path archiveFile = exportDirectoryPath.resolve(de.value.email + ".zip");
		try (OutputStream fOut = Files.newOutputStream(archiveFile);
				BufferedOutputStream bOut = new BufferedOutputStream(fOut);
				ZipOutputStream zipOut = new ZipOutputStream(bOut)) {

			addFileToZip(zipOut, outputDir, de.value.email);
			if (emailOutputDir != null) {
				addFileToZip(zipOut, emailOutputDir, de.value.email + "/email");
			}
		} catch (Exception e) {
			throw new CliException("Error generating archive file", e);
		}
		return archiveFile.toFile();
	}

	private void addFileToZip(ZipOutputStream zipOS, Path path, String base) throws IOException {
		File f = path.toFile();
		if (f.isFile()) {
			zipOS.putNextEntry(new ZipEntry(base));
			try (InputStream fileIS = Files.newInputStream(f.toPath())) {
				byte[] buffer = new byte[BUFFER_SIZE];
				int count = 0;
				while ((count = fileIS.read(buffer)) > 0) {
					zipOS.write(buffer, 0, count);
				}

			}
		} else {
			zipOS.putNextEntry(new ZipEntry(base + "/"));
			File[] children = f.listFiles();
			if (children != null) {
				for (File child : children) {
					addFileToZip(zipOS, child.toPath(), base + "/" + child.getName());
				}
			}
		}
	}

	private void exportData(Path outputDir, ItemValue<Domain> domain, ItemValue<DirEntry> de, String dataType) {
		File outputDataDir = prepareTmpDir(outputDir, dataType);
		try {
			switch (dataType) {
			case "calendar":
				ExportCalendarCommand calendarExportCommand = new ExportCalendarCommand();
				calendarExportCommand.forTarget(de.value.email);
				calendarExportCommand.rootDir = outputDataDir.getAbsolutePath();
				calendarExportCommand.forContext(ctx);
				calendarExportCommand.run();
				break;
			case "contact":
				ExportAddressBookCommand abExportCommand = new ExportAddressBookCommand();
				abExportCommand.forTarget(de.value.email);
				abExportCommand.rootDir = outputDataDir.getAbsolutePath();
				abExportCommand.forContext(ctx);
				abExportCommand.run();
				break;
			case "task":
				ExportTodolistCommand todoExportCommand = new ExportTodolistCommand();
				todoExportCommand.forTarget(de.value.email);
				todoExportCommand.rootDir = outputDataDir.getAbsolutePath();
				todoExportCommand.forContext(ctx);
				todoExportCommand.run();
				break;
			case "notes":
				ExportNotesCommand notesExportCommand = new ExportNotesCommand();
				notesExportCommand.forTarget(de.value.email);
				notesExportCommand.rootDir = outputDataDir.getAbsolutePath();
				notesExportCommand.forContext(ctx);
				notesExportCommand.run();
				break;
			}
		} catch (Exception e) {
			logger.error("error while exporting {}: {}", dataType, e.getMessage(), e);
			throw new CliException("Error while exporting " + dataType, e);
		}
	}

	private Path exportMailData(ItemValue<Domain> domain, ItemValue<DirEntry> de) {
		String returnPath;
		try {
			returnPath = ctx.adminApi().instance(ISdsBackup.class).backupMailbox(domain.uid, de.uid,
					downloadEmailContent);
			ctx.info("mails of " + de.value.email + " has been exported to " + returnPath);
			return Paths.get(returnPath);
		} catch (ServerFault e) {
			logger.error("Cannot archive mails: {}", e.getMessage());
			return null;
		}
	}

	private File prepareTmpDir(Path outputDir, String dataType) {
		File dir = outputDir.resolve(dataType).toFile();
		dir.mkdir();
		return dir;
	}

	@Override
	public Kind[] getDirEntryKind() {
		return new Kind[] { Kind.USER };
	}
}