package net.bluemind.cli.mail;

import java.nio.file.Files;
import java.nio.file.Path;
import java.util.List;
import java.util.Optional;
import java.util.Set;

import net.bluemind.backend.mail.replica.api.IDbMailboxRecords;
import net.bluemind.backend.mail.replica.api.IMailReplicaUids;
import net.bluemind.backend.mail.replica.api.IReplicatedMailboxesMgmt;
import net.bluemind.backend.mail.replica.api.MailboxRecordItemUri;
import net.bluemind.cli.cmd.api.CliContext;
import net.bluemind.cli.cmd.api.ICmdLet;
import net.bluemind.cli.cmd.api.ICmdLetRegistration;
import net.bluemind.core.container.api.IContainers;
import net.bluemind.core.container.model.BaseContainerDescriptor;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.IServiceProvider;
import net.bluemind.mailbox.api.IMailboxes;
import net.bluemind.mailbox.api.Mailbox;
import picocli.CommandLine.Command;
import picocli.CommandLine.Parameters;

@Command(name = "vacuum-bodies", description = "Get rid of references to lost bodies")
public class VacuumBodiesCommand implements ICmdLet, Runnable {

	private CliContext ctx;

	@Parameters(paramLabel = "<file>", description = "One body guid per line")
	public Path file = null;

	@Override
	public void run() {
		try {
			List<String> guids = Files.readAllLines(file);
			ctx.info("Clearing references to {} bodies", guids.size());
			for (String guid : guids) {
				vacuumBody(ctx.adminApi(), guid);
			}
		} catch (Exception e) {
			ctx.error("Error vacuum-ing bodies {}", e.getMessage(), e);
		}
	}

	private void vacuumBody(IServiceProvider serviceProvider, String guid) {
		IReplicatedMailboxesMgmt mgmtApi = serviceProvider.instance(IReplicatedMailboxesMgmt.class);
		Set<MailboxRecordItemUri> refs = mgmtApi.getBodyGuidReferences(guid);
		int cleared = 0;
		for (MailboxRecordItemUri uri : refs) {
			IContainers contApi = serviceProvider.instance(IContainers.class);
			BaseContainerDescriptor cont = contApi.getLightIfPresent(uri.containerUid);
			IMailboxes mboxes = serviceProvider.instance(IMailboxes.class, cont.domainUid);
			ItemValue<Mailbox> mbox = mboxes.getComplete(uri.owner);
			if (mbox == null) {
				ctx.error("Mailbox of {} not found. Deleted?", uri.owner);
			} else {
				IDbMailboxRecords recIntApi = serviceProvider.instance(IDbMailboxRecords.class,
						IMailReplicaUids.uniqueId(uri.containerUid));
				recIntApi.delete(uri.itemUid);
				cleared++;
			}
		}
		ctx.info("Cleared {} reference(s) to {}", cleared, guid);
	}

	@Override
	public Runnable forContext(CliContext ctx) {
		this.ctx = ctx;
		return this;
	}

	public static class Reg implements ICmdLetRegistration {

		@Override
		public Optional<String> group() {
			return Optional.of("mail");
		}

		@Override
		public Class<? extends ICmdLet> commandClass() {
			return VacuumBodiesCommand.class;
		}
	}

}
