/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.eas.command.ping;

import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;

import io.vertx.core.Handler;
import io.vertx.core.eventbus.Message;
import io.vertx.core.eventbus.MessageConsumer;
import io.vertx.core.json.JsonObject;
import net.bluemind.core.caches.registry.CacheRegistry;
import net.bluemind.core.caches.registry.ICacheRegistration;
import net.bluemind.eas.backend.IBackend;
import net.bluemind.eas.dto.EasBusEndpoints;
import net.bluemind.eas.dto.IPreviousRequestsKnowledge;
import net.bluemind.eas.dto.OptionalParams;
import net.bluemind.eas.dto.ping.PingRequest;
import net.bluemind.eas.dto.ping.PingRequest.Folders.Folder;
import net.bluemind.eas.dto.ping.PingResponse;
import net.bluemind.eas.dto.ping.PingResponse.Status;
import net.bluemind.eas.dto.sync.CollectionId;
import net.bluemind.eas.dto.sync.CollectionSyncRequest;
import net.bluemind.eas.impl.Backends;
import net.bluemind.eas.impl.Responder;
import net.bluemind.eas.protocol.IEasProtocol;
import net.bluemind.eas.serdes.ping.PingRequestParser;
import net.bluemind.eas.serdes.ping.PingResponseFormatter;
import net.bluemind.eas.session.BackendSession;
import net.bluemind.eas.session.ItemChangeReference;
import net.bluemind.eas.store.ISyncStorage;
import net.bluemind.eas.utils.EasLogUser;
import net.bluemind.eas.wbxml.builder.WbxmlResponseBuilder;
import net.bluemind.vertx.common.request.Requests;

public class PingProtocol implements IEasProtocol<PingRequest, PingResponse> {
	private static final Logger logger = LoggerFactory.getLogger(PingProtocol.class);
	private static HeartbeatSync heartbeatSync;
	private final ISyncStorage store;
	private final IBackend backend;

	private static final Cache<String, Integer> heartbeat = Caffeine.newBuilder().recordStats().build();

	static {
		heartbeatSync = new HeartbeatSync();
		heartbeatSync.start(heartbeat);
	}

	public static class CacheRegistration implements ICacheRegistration {
		@Override
		public void registerCaches(CacheRegistry cr) {
			cr.register(PingProtocol.class, heartbeat);
		}
	}

	public PingProtocol() {
		backend = Backends.dataAccess();
		store = Backends.internalStorage();
	}

	@Override
	public void parse(BackendSession bs, OptionalParams optParams, Document doc, IPreviousRequestsKnowledge past,
			Handler<PingRequest> parserResultHandler) {
		if (logger.isDebugEnabled()) {
			EasLogUser.logDebugAsUser(bs.getLoginAtDomain(), logger, "******** Parsing *******");
		}

		PingRequestParser parser = new PingRequestParser();
		PingRequest parsed = parser.parse(optParams, doc, past, bs.getLoginAtDomain());
		parserResultHandler.handle(parsed);
	}

	@Override
	public void execute(BackendSession bs, PingRequest query, Handler<PingResponse> pingResponseHandler) {
		if (logger.isDebugEnabled()) {
			EasLogUser.logDebugAsUser(bs.getLoginAtDomain(), logger, "******** Executing *******");
		}

		long intervalSeconds = getIntervalSeconds(query, bs);
		PingResponseHandler responseHandler = handlePingResponseChanges(bs, pingResponseHandler, intervalSeconds);

		PingResponse response = new PingResponse();
		if (query == null) {
			if (bs.getLastMonitored() == null || bs.getLastMonitored().isEmpty()) {
				EasLogUser.logWarnAsUser(bs.getLoginAtDomain(), logger,
						"[{}][{}] Don't know what to monitor, interval: {} toMonitor: {}. Send status 3 MissingParameter",
						bs.getLoginAtDomain(), bs.getDevId(), intervalSeconds, bs.getLastMonitored());
				response.status = Status.MISSING_PARAMETER;
				responseHandler.handle(response);
				return;
			}
			EasLogUser.logInfoAsUser(bs.getLoginAtDomain(), logger,
					"[{}][{}] Empty Ping, reusing cached heartbeat & monitored folders ({})", bs.getLoginAtDomain(),
					bs.getDevId(), bs.getLastMonitored().size());
		} else {
			Set<CollectionSyncRequest> toMonitor = new HashSet<>();
			for (Folder folder : query.folders.folders) {
				try {
					CollectionSyncRequest sc = new CollectionSyncRequest();
					sc.setDataClass(folder.clazz.name());
					sc.setCollectionId(CollectionId.of(folder.id));
					toMonitor.add(sc);
				} catch (NumberFormatException nfe) {
					// HTC ONE X sends "InvalidTaskID" as folder.id
					EasLogUser.logWarnAsUser(bs.getLoginAtDomain(), logger, "[{}][{}] Invalid collectionId {}",
							bs.getLoginAtDomain(), bs.getDevId(), folder.id);
				}

			}
			if (!query.folders.folders.isEmpty()) {
				bs.setLastMonitored(toMonitor);
			}

			// when push list is empty, send MissingParameter
			if (bs.getLastMonitored() == null || bs.getLastMonitored().isEmpty()) {
				EasLogUser.logWarnAsUser(bs.getLoginAtDomain(), logger,
						"[{}][{}]  Nothing to monitor. Send status 3 MissingParameter", bs.getLoginAtDomain(),
						bs.getDevId());
				response.status = Status.MISSING_PARAMETER;
				responseHandler.handle(response);
				return;
			}

			int maxInterval = getInterval("eas_max_heartbeat", 1130);
			if (intervalSeconds > maxInterval) {
				EasLogUser.logWarnAsUser(bs.getLoginAtDomain(), logger,
						"[{}][{}] Send Heartbeat error: intervalSeconds {} > maxInterval {}", bs.getLoginAtDomain(),
						bs.getDevId(), intervalSeconds, maxInterval);
				response.status = Status.INVALID_HEARTBEAT_INTERVAL;
				response.heartbeatInterval = maxInterval;
				responseHandler.handle(response);
				return;
			}

			int minInterval = getInterval("eas_min_heartbeat", 120);
			if (intervalSeconds < minInterval) {
				EasLogUser.logWarnAsUser(bs.getLoginAtDomain(), logger,
						"[{}][{}] Send Heartbeat error: intervalSeconds {} < minInterval {}", bs.getLoginAtDomain(),
						bs.getDevId(), intervalSeconds, minInterval);
				response.status = Status.INVALID_HEARTBEAT_INTERVAL;
				response.heartbeatInterval = minInterval;
				responseHandler.handle(response);
				return;
			}

			if (intervalSeconds != bs.getHeartbeart()) {
				bs.setHeartbeart(intervalSeconds);
				store.updateLastHearbeat(bs.getDeviceId(), intervalSeconds);
			}

			// Samsung devices with samsung apps might not process MoreAvailable elements
			// and will rely on Ping to fetch more data
			for (CollectionSyncRequest watched : toMonitor) {
				Queue<ItemChangeReference> pending = bs.getUnSynchronizedItemChange(watched.getCollectionId());
				if (pending != null && !pending.isEmpty()) {
					EasLogUser.logInfoAsUser(bs.getLoginAtDomain(), logger,
							"[{}][{}] Instant wake-up for pending changes to {}", bs.getLoginAtDomain(), bs.getDevId(),
							watched.getCollectionId());
					response.status = Status.CHANGES_OCCURRED;
					response.folders = new PingResponse.Folders();
					response.folders.folders.add(watched.getCollectionId().getValue());
					responseHandler.handle(response);
					return;
				}
			}

		}

	}

	private PingResponseHandler handlePingResponseChanges(BackendSession bs, Handler<PingResponse> prHandler,
			long intervalSeconds) {

		PingResponse response = new PingResponse();
		PingResponseHandler responseHandler = new PingResponseHandler(prHandler, bs);

		if (intervalSeconds > 0 && bs.getLastMonitored() != null) {
			Requests.tagAsync(bs.getRequest());
			Requests.tag(bs.getRequest(), "timeout", intervalSeconds + "s");

			final List<MessageConsumer<JsonObject>> consumers = new LinkedList<>();
			final AtomicBoolean responseSent = new AtomicBoolean();
			long noChangesTimer = responseHandler.vertx.setTimer(TimeUnit.SECONDS.toMillis(intervalSeconds), tid -> {
				// noChanges
				if (responseSent.getAndSet(true)) {
					return;
				}
				consumers.forEach(MessageConsumer::unregister);
				responseHandler.handle(PingResponse.createNoChangesPingResponse());
			});

			responseHandler.registerPushKiller(noChangesTimer, responseSent);

			for (CollectionSyncRequest sc : bs.getLastMonitored()) {
				handleCollectionChanges(responseHandler, consumers, responseSent, noChangesTimer, sc);
			}

			handleHierarchyChanges(bs, responseHandler, consumers, responseSent, noChangesTimer);

			handleResetRequets(bs, responseHandler, consumers, responseSent, noChangesTimer);

			handleDelegateChanges(bs, responseHandler, consumers, responseSent, noChangesTimer);

		} else {
			EasLogUser.logWarnAsUser(bs.getLoginAtDomain(), logger,
					"[{}][{}] Don't know what to monitor, interval is null. Send status 3 MissingParameter",
					bs.getLoginAtDomain(), bs.getDevId());
			response.status = Status.MISSING_PARAMETER;
			responseHandler.handle(response);
		}

		return responseHandler;
	}

	private long getIntervalSeconds(PingRequest query, BackendSession bs) {
		long interval = getLastHeartbeat(bs);

		if (query != null && query.heartbeatInterval != null) {
			interval = query.heartbeatInterval;
		}

		return interval;
	}

	private void handleCollectionChanges(PingResponseHandler responseHandler,
			final List<MessageConsumer<JsonObject>> consumers, final AtomicBoolean responseSent, long noChangesTimer,
			CollectionSyncRequest sc) {
		MessageConsumer<JsonObject> cons = responseHandler.eventBus
				.consumer(EasBusEndpoints.SYNC_COLLECTION + sc.getCollectionId().getFolderId());
		consumers.add(cons);
		Handler<Message<JsonObject>> colChangeHandler = (Message<JsonObject> msg) -> {
			// syncRequired
			if (responseSent.getAndSet(true)) {
				return;
			}
			consumers.forEach(MessageConsumer::unregister);
			responseHandler.vertx.cancelTimer(noChangesTimer);
			PingResponse pr = new PingResponse();
			pr.status = Status.CHANGES_OCCURRED;
			pr.folders = new PingResponse.Folders();
			pr.folders.folders.add(sc.getCollectionId().getValue());
			responseHandler.handle(pr);
		};
		cons.handler(colChangeHandler);
	}

	private void handleDelegateChanges(BackendSession bs, PingResponseHandler responseHandler,
			final List<MessageConsumer<JsonObject>> consumers, final AtomicBoolean responseSent, long noChangesTimer) {
		backend.getSubscriptions(bs).forEach(owner -> {
			consumeMsgAndAskFolderSync(responseHandler, consumers, responseSent, noChangesTimer,
					EasBusEndpoints.SYNC_DELEGATION + owner);
			consumeMsgAndAskFolderSync(responseHandler, consumers, responseSent, noChangesTimer,
					EasBusEndpoints.SYNC_HIERARCHY + owner);
		});
	}

	private void handleResetRequets(BackendSession bs, PingResponseHandler responseHandler,
			final List<MessageConsumer<JsonObject>> consumers, final AtomicBoolean responseSent, long noChangesTimer) {
		String address = EasBusEndpoints.SYNC_RESET + bs.getUser().getUid() + "#" + bs.getDevId();
		consumeMsgAndAskFolderSync(responseHandler, consumers, responseSent, noChangesTimer, address);
	}

	private void handleHierarchyChanges(BackendSession bs, PingResponseHandler responseHandler,
			final List<MessageConsumer<JsonObject>> consumers, final AtomicBoolean responseSent, long noChangesTimer) {
		consumeMsgAndAskFolderSync(responseHandler, consumers, responseSent, noChangesTimer,
				EasBusEndpoints.SYNC_HIERARCHY + bs.getUser().getUid());
	}

	private void consumeMsgAndAskFolderSync(PingResponseHandler responseHandler,
			final List<MessageConsumer<JsonObject>> consumers, final AtomicBoolean responseSent, long noChangesTimer,
			String busName) {
		MessageConsumer<JsonObject> msgConsumer = responseHandler.eventBus.consumer(busName);
		consumers.add(msgConsumer);
		Handler<Message<JsonObject>> msgChangeHandler = (Message<JsonObject> msg) -> {
			if (responseSent.getAndSet(true)) {
				return;
			}
			consumers.forEach(MessageConsumer::unregister);
			responseHandler.vertx.cancelTimer(noChangesTimer);
			PingResponse pr = new PingResponse();
			pr.status = Status.FOLDER_SYNC_REQUIRED;
			responseHandler.handle(pr);
		};
		msgConsumer.handler(msgChangeHandler);
	}

	private long getLastHeartbeat(BackendSession bs) {
		Long hb = bs.getHeartbeart();
		if (hb == null) {
			hb = store.findLastHeartbeat(bs.getDeviceId());
			bs.setHeartbeart(hb);
		}
		return hb;
	}

	private int getInterval(String k, int defaultValue) {
		Integer ret = heartbeat.getIfPresent(k);
		if (ret == null) {
			ret = defaultValue;
			String val = store.getSystemConf(k);
			if (val != null) {
				try {
					ret = Integer.parseInt(val);
				} catch (NumberFormatException nfe) {
					logger.error("Invalid {} value {} ", k, val);
				}
			}
			heartbeat.put(k, ret);
		}

		return ret;
	}

	@Override
	public void write(BackendSession bs, Responder responder, PingResponse response, final Handler<Void> completion) {
		PingResponseFormatter formatter = new PingResponseFormatter();
		WbxmlResponseBuilder builder = new WbxmlResponseBuilder(bs, responder.asOutput());
		formatter.format(builder, bs.getProtocolVersion(), response, data -> completion.handle(data));
	}

	@Override
	public String address() {
		return "eas.protocol.ping";
	}

}
