/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2016
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.eas.command.meetingresponse;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Optional;
import java.util.concurrent.atomic.AtomicInteger;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;

import io.vertx.core.Handler;
import net.bluemind.eas.backend.HierarchyNode;
import net.bluemind.eas.backend.IBackend;
import net.bluemind.eas.backend.dto.CollectionIdContext;
import net.bluemind.eas.backend.importer.ContentImportEntityForDeletion;
import net.bluemind.eas.backend.importer.IContentsImporter;
import net.bluemind.eas.data.calendarenum.AttendeeStatus;
import net.bluemind.eas.dto.IPreviousRequestsKnowledge;
import net.bluemind.eas.dto.OptionalParams;
import net.bluemind.eas.dto.base.AppData;
import net.bluemind.eas.dto.base.CollectionItem;
import net.bluemind.eas.dto.calendar.CalendarResponse;
import net.bluemind.eas.dto.calendar.CalendarResponse.InstanceType;
import net.bluemind.eas.dto.meetingresponse.MeetingResponseRequest;
import net.bluemind.eas.dto.meetingresponse.MeetingResponseRequest.Request;
import net.bluemind.eas.dto.meetingresponse.MeetingResponseResponse;
import net.bluemind.eas.dto.meetingresponse.MeetingResponseResponse.Result.Status;
import net.bluemind.eas.dto.sync.CollectionId;
import net.bluemind.eas.dto.type.ItemDataType;
import net.bluemind.eas.exception.ActiveSyncException;
import net.bluemind.eas.impl.Backends;
import net.bluemind.eas.impl.Responder;
import net.bluemind.eas.impl.vertx.VertxLazyLoader;
import net.bluemind.eas.protocol.IEasProtocol;
import net.bluemind.eas.serdes.IResponseBuilder;
import net.bluemind.eas.serdes.meetingresponse.MeetingResponseRequestParser;
import net.bluemind.eas.serdes.meetingresponse.MeetingResponseResponseFormatter;
import net.bluemind.eas.session.BackendSession;
import net.bluemind.eas.session.ItemChangeReference;
import net.bluemind.eas.store.ISyncStorage;
import net.bluemind.eas.utils.EasLogUser;
import net.bluemind.eas.wbxml.builder.WbxmlResponseBuilder;

public class MeetingResponseProtocol implements IEasProtocol<MeetingResponseRequest, MeetingResponseResponse> {

	private static final Logger logger = LoggerFactory.getLogger(MeetingResponseProtocol.class);

	private final IBackend backend;
	private final ISyncStorage store;

	public MeetingResponseProtocol() {
		backend = Backends.dataAccess();
		store = Backends.internalStorage();
	}

	@Override
	public void parse(BackendSession bs, OptionalParams optParams, Document doc, IPreviousRequestsKnowledge past,
			Handler<MeetingResponseRequest> parserResultHandler) {
		MeetingResponseRequestParser parser = new MeetingResponseRequestParser();
		MeetingResponseRequest parsed = parser.parse(optParams, doc, past, bs.getLoginAtDomain());
		parserResultHandler.handle(parsed);
	}

	@Override
	public void execute(final BackendSession bs, MeetingResponseRequest query,
			final Handler<MeetingResponseResponse> responseHandler) {
		final MeetingResponseResponse response = new MeetingResponseResponse();
		response.results = new ArrayList<MeetingResponseResponse.Result>(query.requests.size());

		final AtomicInteger toProcess = new AtomicInteger(query.requests.size());
		for (final Request request : query.requests) {
			try {
				HierarchyNode node = store
						.getHierarchyNode(new CollectionIdContext(bs, CollectionId.of(request.collectionId)));
				ItemDataType dataClass = ItemDataType.getValue(node.containerType);

				ItemChangeReference ic = new ItemChangeReference(dataClass);
				ic.setServerId(CollectionItem.of(sanitizeRequestId(request.requestId)));

				final MeetingResponseResponse.Result r = new MeetingResponseResponse.Result();
				final ItemChangeReference itemRef = ic;
				// The RequestId element is present in MeetingResponse
				// command responses only if it was present in the
				// corresponding MeetingResponse command request. The
				// RequestId element MUST NOT be present in the
				// MeetingResponse command request if the search:LongId
				// element is present.
				if (request.requestId != null && request.LongId == null) {
					r.requestId = request.requestId;
				}

				invitation(bs, itemRef, invitation -> {
					if (invitation == null) {
						EasLogUser.logErrorAsUser(bs.getLoginAtDomain(), logger, "Invalid meeting request for {}",
								r.requestId);
						r.status = Status.INVALID_MEETING_REQUEST;
					} else {
						if ((invitation.instanceType == InstanceType.SINGLE_INSTANCE
								|| invitation.instanceType == InstanceType.EXCEPTION_TO_RECURRING)
								&& invitation.recurrenceId != null) {
							request.instanceId = invitation.recurrenceId;
						}
						r.status = Status.SUCCESS;

						AttendeeStatus attendeeStatus = null;
						switch (request.userResponse) {
						case Accepted:
							attendeeStatus = AttendeeStatus.ACCEPT;
							break;
						case Declined:
							attendeeStatus = AttendeeStatus.DECLINE;
							break;
						case TentativelyAccepted:
							attendeeStatus = AttendeeStatus.TENTATIVE;
							break;
						default:
							break;
						}

						long itemId = itemRef.getServerId().itemId;
						if (itemRef.getType() == ItemDataType.EMAIL) {
							itemId = invitation.itemUid;
						}
						IContentsImporter importer = backend.getContentsImporter(bs);
						String calendarId = importer.importCalendarUserStatus(bs, itemId, attendeeStatus,
								request.instanceId, invitation.calendarUid);

						// 2.2.3.18 CalendarId
						// If the meeting is declined, the response does not
						// contain a CalendarId element.
						if (attendeeStatus != AttendeeStatus.DECLINE) {
							r.calendarId = calendarId;
						}

						// delete the email
						deleteMeetingRequest(bs, itemRef);
					}

					response.results.add(r);
					subRequestProcessed(toProcess, responseHandler, response);
				});

			} catch (Exception e) {
				EasLogUser.logExceptionAsUser(bs.getLoginAtDomain(), e, logger);
				MeetingResponseResponse.Result r = new MeetingResponseResponse.Result();
				r.requestId = request.requestId;
				r.status = Status.SERVER_ERROR;
				response.results.add(r);
				subRequestProcessed(toProcess, responseHandler, response);
			}

		}
	}

	private String sanitizeRequestId(String requestId) {
		// iOS sometimes sends
		// collectionId:itemId&lt;!ExceptionDate!&gt;20010101T000000Z as requestId
		// collectionId:itemId is expected
		if (requestId.contains("&lt;!ExceptionDate!&gt;")) {
			requestId = requestId.substring(0, requestId.indexOf("&lt;!ExceptionDate!&gt;"));
		}
		return requestId;

	}

	private void subRequestProcessed(AtomicInteger toProcess, Handler<MeetingResponseResponse> responseHandler,
			MeetingResponseResponse response) {
		int now = toProcess.decrementAndGet();
		if (now == 0) {
			responseHandler.handle(response);
		}
	}

	private void invitation(BackendSession bs, ItemChangeReference ic, Handler<CalendarResponse> foundInvite) {
		Optional<AppData> optData = ic.getData();
		AppData loaded = null;
		if (!optData.isPresent()) {
			try {
				loaded = backend.getContentsExporter(bs).loadStructure(bs, null, ic);
				loaded.body = VertxLazyLoader.wrap(loaded.body);
			} catch (ActiveSyncException e) {
				EasLogUser.logExceptionAsUser(bs.getLoginAtDomain(), e, logger);
				foundInvite.handle(null);
				return;
			}
		} else {
			loaded = optData.get();
		}
		CalendarResponse cr = null;
		if (ic.getType() == ItemDataType.EMAIL) {
			if (loaded.metadata.email != null) {

				cr = loaded.metadata.email.meetingRequest;
				cr.calendarUid = loaded.metadata.email.calendarUid;
				EasLogUser.logDebugAsUser(bs.getLoginAtDomain(), logger, "Loaded invitation from email {}", cr);
			}
		} else if (ic.getType() == ItemDataType.CALENDAR) {
			cr = loaded.metadata.event;
			EasLogUser.logDebugAsUser(bs.getLoginAtDomain(), logger, "Loaded invitation from calendar {}", cr);
		}
		foundInvite.handle(cr);
	}

	private void deleteMeetingRequest(BackendSession bs, ItemChangeReference ic) {
		if (ic.getType() != ItemDataType.EMAIL) {
			EasLogUser.logInfoAsUser(bs.getLoginAtDomain(), logger, "Can't delete meeting request for type {}",
					ic.getType());
			return;
		}
		try {
			IContentsImporter mailImporter = backend.getContentsImporter(bs);
			mailImporter.importMessageDeletion(ContentImportEntityForDeletion.create(bs, ItemDataType.EMAIL,
					Arrays.asList(ic.getServerId()), false));
		} catch (Exception t) {
			EasLogUser.logExceptionAsUser(bs.getLoginAtDomain(), t, logger);
		}
	}

	@Override
	public void write(BackendSession bs, Responder responder, MeetingResponseResponse response,
			final Handler<Void> completion) {
		MeetingResponseResponseFormatter formatter = new MeetingResponseResponseFormatter();
		IResponseBuilder builder = new WbxmlResponseBuilder(bs, responder.asOutput());
		formatter.format(builder, bs.getProtocolVersion(), response, data -> completion.handle(null));
	}

	@Override
	public String address() {
		return "eas.protocol.meetingresponse";
	}

}
