/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2023
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */

package net.bluemind.core.auditlogs.client.es.datastreams;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;

import org.eclipse.core.runtime.IConfigurationElement;
import org.eclipse.core.runtime.IExtension;
import org.eclipse.core.runtime.IExtensionPoint;
import org.eclipse.core.runtime.Platform;
import org.osgi.framework.Bundle;
import org.osgi.framework.BundleActivator;
import org.osgi.framework.BundleContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.io.ByteStreams;

public class DataStreamActivator implements BundleActivator {

	static final IndexTemplateDefinition indexTemplateDefinition = loadIndexTemplateDefinition();
	static final ILMPolicyDefinition ilmPolicyDefinition = loadILMPolicyDefinition();

	private static Logger logger = LoggerFactory.getLogger(DataStreamActivator.class);

	@Override
	public void start(BundleContext arg0) throws Exception {
		/// Nothing to do
	}

	@Override
	public void stop(BundleContext arg0) throws Exception {
		/// Nothing to do
	}

	private static IndexTemplateDefinition loadIndexTemplateDefinition() {
		IExtensionPoint ep = Platform.getExtensionRegistry()
				.getExtensionPoint("net.bluemind.core.auditlogs.client.es.datastreams", "indextemplate");
		for (IExtension ext : ep.getExtensions()) {
			for (IConfigurationElement ce : ext.getConfigurationElements()) {
				String indexTemplateName = ce.getAttribute("indextemplatename");
				String schema = ce.getAttribute("resource");
				Bundle bundle = Platform.getBundle(ext.getContributor().getName());
				URL url = bundle.getResource(schema);
				try (InputStream in = url.openStream()) {
					return new IndexTemplateDefinition(indexTemplateName, ByteStreams.toByteArray(in));
				} catch (IOException e) {
					logger.error("Cannot open audit log configuration file '{}'", url.getFile());
					return null;
				}
			}
		}
		return null;
	}

	private static ILMPolicyDefinition loadILMPolicyDefinition() {
		IExtensionPoint ep = Platform.getExtensionRegistry()
				.getExtensionPoint("net.bluemind.core.auditlogs.client.es.datastreams", "ilmpolicy");
		for (IExtension ext : ep.getExtensions()) {
			for (IConfigurationElement ce : ext.getConfigurationElements()) {
				String ilmPolicyName = ce.getAttribute("name");
				String schema = ce.getAttribute("resource");
				Bundle bundle = Platform.getBundle(ext.getContributor().getName());
				URL url = bundle.getResource(schema);
				try (InputStream in = url.openStream()) {
					return new ILMPolicyDefinition(ilmPolicyName, ByteStreams.toByteArray(in));
				} catch (IOException e) {
					logger.error("Cannot open audit log ILM policy file '{}'", url.getFile());
					return null;
				}
			}
		}
		return null;
	}

	record IndexTemplateDefinition(String indexTemplateName, byte[] schema) {

	}

	record ILMPolicyDefinition(String name, byte[] schema) {

	}

	static IndexTemplateDefinition getindexTemplateDefinition() {
		return indexTemplateDefinition;
	}

	static ILMPolicyDefinition getIlmPolicyDefinition() {
		return ilmPolicyDefinition;
	}

}