package net.bluemind.central.reverse.proxy.model.common.mapper.impl;

import java.io.IOException;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.JsonToken;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;

import net.bluemind.central.reverse.proxy.model.common.mapper.RecordKey;
import net.bluemind.central.reverse.proxy.model.common.mapper.RecordKeyMapper;

public class ByteArrayRecordKeyMapper implements RecordKeyMapper<byte[]> {

	private final Logger logger = LoggerFactory.getLogger(ByteArrayRecordKeyMapper.class);
	private final JsonFactory jsonFactory;
	private final ObjectWriter writer;

	public ByteArrayRecordKeyMapper(ObjectMapper objectMapper) {
		jsonFactory = objectMapper.getFactory();
		this.writer = objectMapper.writerFor(RecordKey.class);
	}

	public Optional<RecordKey> map(byte[] keyBytes) {
		String type = null;
		String owner = null;
		String uid = null;
		long id = 0;
		String valueClass = null;
		String operation = null;
		try (JsonParser parser = jsonFactory.createParser(keyBytes)) {
			JsonToken tok;
			if (parser.nextToken() != JsonToken.START_OBJECT) {
				logger.error("Error parsing RecordKey");
				return Optional.empty();
			}
			while ((tok = parser.nextToken()) != JsonToken.END_OBJECT && tok != null) {
				String fieldName = parser.currentName();
				parser.nextToken();
				switch (fieldName) {
				case "type" -> type = parser.getText();
				case "owner" -> owner = parser.getText();
				case "uid" -> uid = parser.getText();
				case "id" -> id = parser.getLongValue();
				case "valueClass" -> valueClass = parser.getText();
				case "operation" -> operation = parser.getText();
				default -> parser.skipChildren();
				}
			}
		} catch (IOException e) {
			logger.error("Error parsing RecordKey", e);
			return Optional.empty();
		}

		return Optional.of(new RecordKey(type, owner, uid, id, valueClass, operation));
	}

	public Optional<byte[]> map(RecordKey key) {
		try {
			return Optional.of(writer.writeValueAsBytes(key));
		} catch (JsonProcessingException jse) {
			logger.error("Unable to serialize key {}", key, jse);
			return Optional.empty();
		}
	}
}
