/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.cli.inject.imap;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.TimeUnit;

import org.columba.ristretto.message.Address;
import org.columba.ristretto.smtp.SMTPProtocol;
import org.columba.ristretto.smtp.SMTPResponse;
import org.slf4j.event.Level;
import org.slf4j.helpers.MessageFormatter;

import com.google.common.collect.Iterators;
import com.google.common.util.concurrent.RateLimiter;

import io.netty.util.internal.ThreadLocalRandom;
import net.bluemind.authentication.api.IAuthentication;
import net.bluemind.authentication.api.LoginResponse;
import net.bluemind.cli.cmd.api.CliContext;
import net.bluemind.cli.cmd.api.DomainNames;
import net.bluemind.cli.cmd.api.ICmdLet;
import net.bluemind.cli.cmd.api.ICmdLetRegistration;
import net.bluemind.cli.utils.CliUtils;
import net.bluemind.config.Token;
import net.bluemind.core.api.Email;
import net.bluemind.core.api.ListResult;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.IServiceProvider;
import net.bluemind.directory.api.BaseDirEntry.Kind;
import net.bluemind.directory.api.DirEntry;
import net.bluemind.directory.api.DirEntryQuery;
import net.bluemind.directory.api.DirEntryQuery.StateFilter;
import net.bluemind.directory.api.IDirectory;
import net.bluemind.domain.api.Domain;
import net.bluemind.group.api.Group;
import net.bluemind.group.api.IGroup;
import net.bluemind.group.api.Member;
import net.bluemind.imap.vt.StoreClient;
import net.bluemind.imap.vt.dto.IdleContext;
import net.bluemind.imap.vt.dto.IdleListener.IdleEvent;
import net.bluemind.imap.vt.dto.Mode;
import net.bluemind.imap.vt.dto.UidFetched;
import net.bluemind.mailbox.api.IMailboxes;
import net.bluemind.mailbox.api.Mailbox;
import picocli.CommandLine.Command;
import picocli.CommandLine.Option;
import picocli.CommandLine.Parameters;

@Command(name = "imap-outlook", description = "Simulate Outlook send/receive loops")
public class ImapOutlookStyleInjectCommand implements ICmdLet, Runnable {

	@Option(names = "--connections", description = "Concurrent connections")
	public int connections = 64;

	@Option(names = "--loops", description = "Send & Receive loops")
	public int loops = 256;

	@Option(names = "--faults-per-sec", description = "Inject X random faults every seconds")
	public int faultPerSec = 0;

	@Parameters(paramLabel = "<domain_name>", description = "the domain (uid or alias)", completionCandidates = DomainNames.class)
	public String domain;

	@Option(names = "--imap-host", description = "IMAP endpoint")
	public String endpoint = "127.0.0.1";

	@Option(names = "--imap-port", description = "IMAP endpoint")
	public int port = 143;

	public static class Reg implements ICmdLetRegistration {

		@Override
		public Optional<String> group() {
			return Optional.of("inject");
		}

		@Override
		public Class<? extends ICmdLet> commandClass() {
			return ImapOutlookStyleInjectCommand.class;
		}
	}

	private CliContext ctx;

	private static record LoopFeedback(Level lvl, String msg) {

		static LoopFeedback of(String p, Object... params) {
			return new LoopFeedback(Level.INFO, MessageFormatter.arrayFormat(p, params).getMessage());
		}

	}

	public static class FaultsTrigger {

		private final RateLimiter rateLimiter;

		public FaultsTrigger(int fps) {
			this.rateLimiter = fps > 0 ? RateLimiter.create(1.0 / fps) : null;
		}

		public boolean shouldFault() {
			return rateLimiter == null ? false : rateLimiter.tryAcquire();
		}

	}

	@Override
	public void run() {

		CliUtils cu = new CliUtils(ctx);
		ItemValue<Domain> dom = cu.getDomain(domain).orElseThrow();
		IServiceProvider prov = ctx.longRequestTimeoutAdminApi();
		IDirectory dirApi = prov.instance(IDirectory.class, dom.uid);
		DirEntryQuery q = DirEntryQuery.filterKind(Kind.USER);
		q.stateFilter = StateFilter.Active;
		q.size = Math.max(connections, 4);
		ListResult<ItemValue<DirEntry>> avail = dirApi.search(q);
		Iterator<ItemValue<DirEntry>> userIter = Iterators.cycle(avail.values);
		Iterator<ItemValue<DirEntry>> memberIter = Iterators.cycle(List.copyOf(avail.values));

		String grpLocal = "delivery.grp" + System.nanoTime();
		String grpEmail = setupDeliveryGroup(dom, prov, memberIter, grpLocal);
		Thread deliverVt = startDeliveryVT(dom, grpEmail);

		ArrayBlockingQueue<LoopFeedback> feedback = new ArrayBlockingQueue<>(30);

		FaultsTrigger faults = new FaultsTrigger(faultPerSec);

		List<Thread> outlookVts = new ArrayList<>();
		RateLimiter idleLogs = RateLimiter.create(1);
		for (int i = 0; i < connections; i++) {
			ItemValue<DirEntry> user = userIter.next();
			IAuthentication authApi = ctx.adminApi().instance(IAuthentication.class);
			IMailboxes mboxApi = ctx.adminApi().instance(IMailboxes.class, dom.uid);
			ItemValue<Mailbox> mbox = mboxApi.getComplete(user.uid);
			String login = mbox.value.name + "@" + dom.uid;
			LoginResponse lr = authApi.su(login);
			if (lr.authKey != null) {
				outlookVts.add(Thread.ofVirtual().name(login + ":" + i)
						.start(() -> outlookSendAndReceive(login, lr, feedback, faults, idleLogs)));
			}
		}
		Thread.ofPlatform().start(() -> pumpFeedback(feedback));
		for (var vt : outlookVts) {
			try {
				vt.join();
				ctx.info("{} finished.", vt);
			} catch (InterruptedException e) {
				Thread.currentThread().interrupt();
			}
		}
		deliverVt.interrupt();
	}

	private String setupDeliveryGroup(ItemValue<Domain> dom, IServiceProvider prov,
			Iterator<ItemValue<DirEntry>> memberIter, String grpLocal) {
		String grpEmail = grpLocal + "@" + dom.value.defaultAlias;
		IGroup grpApi = prov.instance(IGroup.class, dom.uid);
		Group g = new Group();
		g.emails = List.of(Email.create(grpEmail, true));
		g.name = grpLocal;
		grpApi.create(grpLocal, g);
		List<Member> toAdd = new ArrayList<>();
		Set<String> seenUids = new HashSet<>();
		for (int i = 0; i < connections; i++) {
			ItemValue<DirEntry> dirEntry = memberIter.next();
			if (!seenUids.contains(dirEntry.uid)) {
				toAdd.add(Member.user(dirEntry.uid));
				seenUids.add(dirEntry.uid);
			}
		}
		grpApi.add(grpLocal, toAdd);
		ctx.info("Added " + toAdd.size() + " member(s) to " + grpEmail);
		return grpEmail;
	}

	private Thread startDeliveryVT(ItemValue<Domain> dom, String grpEmail) {
		return Thread.ofVirtual().name("deliver-to-" + grpEmail).start(() -> {
			boolean done = false;
			// wait a bit for postfix maps updates
			try {
				Thread.sleep(20000);
			} catch (InterruptedException e) {
				Thread.currentThread().interrupt();
			}
			while (!done) {
				try (SMTPProtocol prot = new SMTPProtocol(endpoint, 587)) {
					Thread.sleep(1000);
					prot.openPort();
					prot.startTLS();
					prot.auth("PLAIN", "admin0@global.virt", Token.admin0().toCharArray());
					prot.mail(Address.parse("noreply@" + dom.value.defaultAlias));
					prot.rcpt(Address.parse(grpEmail));
					SMTPResponse resp = prot
							.data(new ByteArrayInputStream(("Empty-Crap: " + UUID.randomUUID().toString()).getBytes()));
					System.err.println("SEND: " + resp.getCode() + " " + resp.getMessage());
				} catch (InterruptedException e) {
					Thread.currentThread().interrupt();
					done = true;
				} catch (Exception e) {
					System.err.println("Delivery pb: " + e.getMessage());
				}
			}
			System.err.println("Leaving delivery VT");
		});
	}

	private void pumpFeedback(ArrayBlockingQueue<LoopFeedback> feedback) {
		boolean done = false;
		do {
			try {
				LoopFeedback log;
				log = feedback.poll(1, TimeUnit.SECONDS);
				if (log != null) {
					ctx.info(log.msg());
				}
			} catch (InterruptedException e) {
				Thread.currentThread().interrupt();
				done = true;
			}
		} while (!done);
	}

	private void outlookSendAndReceive(String login, LoginResponse lr, ArrayBlockingQueue<LoopFeedback> feedback,
			FaultsTrigger faults, RateLimiter idleLogs) {
		ThreadLocalRandom tr = ThreadLocalRandom.current();
		try (StoreClient sc = new StoreClient(endpoint, port, login, lr.authKey)) {
			if (!sc.login()) {
				return;
			}
			for (int l = 0; l < loops; l++) {
				net.bluemind.imap.vt.dto.ListResult folders = sc.list("", "*");
				for (var li : folders) {
					if (li.isSelectable()) {
						sc.select(li.getName());
						boolean inbox = "inbox".equalsIgnoreCase(li.getName());
						int waitMs = inbox ? 50 : 0;
						idleQuick(sc, idleLogs, waitMs);
						if (faults.shouldFault()) {
							sc.disconnectOnNextChunk();
						}
						List<UidFetched> msgs = sc.uidFetchHeaders("1:*");
						idleQuick(sc, idleLogs, waitMs);
						if (inbox && !msgs.isEmpty()) {
							int toDel = msgs.get(tr.nextInt(msgs.size())).uid();
							sc.uidStore("" + toDel, Mode.SET, "\\Seen", "\\Deleted");
							sc.expunge();
						}
					}
				}
				pauseAndReportProgress(login, feedback, l);
			}
		} catch (IOException e) {
			System.err.println("ERROR occured: " + e.getMessage());
		} catch (InterruptedException e) {
			Thread.currentThread().interrupt();
		}
	}

	private void pauseAndReportProgress(String login, ArrayBlockingQueue<LoopFeedback> feedback, int l)
			throws InterruptedException {
		Thread.sleep(250);
		if (l % 10 == 0) {
			boolean underPressure = !feedback.offer(LoopFeedback.of("[{}] Completed loop {}", login, l), 1,
					TimeUnit.SECONDS);
			if (underPressure) {
				Thread.sleep(250);
			}
		}
	}

	private void idleQuick(StoreClient sc, RateLimiter idleLogs, int waitMs) throws IOException {
		String tn = Thread.currentThread().getName();
		IdleContext idleCtx = sc.idle((IdleContext idle, IdleEvent event) -> {
			if (idleLogs.tryAcquire()) {
				System.err.println("IDLE{" + tn + "} " + event.payload());
			}
		});
		if (waitMs > 0) {
			try {
				Thread.sleep(waitMs);
			} catch (InterruptedException e) {
				Thread.currentThread().interrupt();
				return;
			}
		}
		idleCtx.done();
		idleCtx.join();
	}

	@Override
	public Runnable forContext(CliContext ctx) {
		this.ctx = ctx;
		return this;
	}

}
