/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.addressbook.impl;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.Collections;
import java.util.List;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.addressbook.adapter.VCardAdapter;
import net.bluemind.addressbook.api.VCard;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.task.service.BlockingServerTask;
import net.bluemind.core.task.service.IServerTaskMonitor;
import net.bluemind.dataprotect.api.DataProtectGeneration;
import net.bluemind.dataprotect.api.PartGeneration;
import net.bluemind.dataprotect.api.Restorable;
import net.bluemind.dataprotect.api.RestorableKind;
import net.bluemind.dataprotect.api.WorkerDataType;
import net.bluemind.dataprotect.common.backup.RestorableContainerItemBackupItem;
import net.bluemind.dataprotect.common.restore.IMonitoredRestoreRestorableItem;
import net.bluemind.dataprotect.common.restore.RestoreRestorableItem;
import net.bluemind.dataprotect.service.BackupPath;
import net.bluemind.network.topology.Topology;
import net.bluemind.server.api.Server;
import net.bluemind.server.api.TagDescriptor;

public class AbstractRestoreBook extends BlockingServerTask {
	private static final Logger logger = LoggerFactory.getLogger(AbstractRestoreBook.class);

	protected final DataProtectGeneration backup;
	protected IMonitoredRestoreRestorableItem restorableItem;

	public AbstractRestoreBook(DataProtectGeneration backup, Restorable item) {
		this.backup = backup;
		this.restorableItem = new RestoreRestorableItem(item);
	}

	protected AddressBookBackupRepository getRepository() {
		try {
			List<PartGeneration> parts = backup.parts;
			PartGeneration corepart = parts.stream().filter(p -> WorkerDataType.ADDRESSBOOK == p.getWorkerDataType())
					.findFirst().orElseThrow(() -> ServerFault.notFound("Unable to find backup part 'addressbook'"));
			ItemValue<Server> coreServer = Topology.get().core();
			Path backupPath = Paths.get(BackupPath.get(coreServer, TagDescriptor.bm_core.getTag()),
					String.valueOf(corepart.id), "var/backups/bluemind/addressbooks");
			if (Files.exists(backupPath)) {
				return new AddressBookBackupRepository(backupPath);
			}
		} catch (Exception e) {
			logger.info(
					"Unable to get addressbook backup repository: {}, this message can be ignored if the restore process is related to a legacy database backup",
					e.getMessage());
		}
		return null;
	}

	protected List<ItemValue<VCard>> importVcf(RestorableContainerItemBackupItem restorable, String owner,
			String domain, RestorableKind ownerKind) throws IOException {
		return Optional.ofNullable(new String(restorable.dataStream().readAllBytes(), StandardCharsets.UTF_8))
				.filter(vcard -> !vcard.isBlank())
				.map(vcard -> VCardAdapter.parseBmVCards(vcard, domain, owner, RestorableKind.getKind(ownerKind)))
				.orElse(Collections.emptyList());
	}

	@Override
	protected void run(IServerTaskMonitor monitor) throws Exception {
		throw new UnsupportedOperationException("not implemented");
	}
}
