/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2023
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */

package net.bluemind.core.auditlogs;

import java.io.IOException;

import jakarta.ws.rs.GET;
import jakarta.ws.rs.POST;
import jakarta.ws.rs.Path;
import net.bluemind.core.api.BMApi;
import net.bluemind.core.auditlogs.exception.AuditLogCreationException;
import net.bluemind.core.auditlogs.exception.AuditLogILMPolicyException;
import net.bluemind.core.auditlogs.exception.AuditLogRemovalException;

@BMApi(version = "3", internal = true)
@Path("/logs/manage")
public interface IAuditLogMgmt {

	/*
	 * Creates a datastream using datastream name pattern defined in
	 * auditlog-store.conf file
	 *
	 * @param domainUid the domain uid for the auditlog datastream to create
	 * 
	 * @throws AuditLogCreationException
	 */
	@POST
	@Path("_create_auditlog")
	public void setupAuditLogBackingStore(String domainUid) throws AuditLogCreationException;

	/*
	 * Removes all auditlog datastreams
	 *
	 * @throws AuditLogCreationException
	 */
	@POST
	@Path("_remove_all")
	public void removeAuditLogBackingStores();

	/*
	 * Removes a datastream using datastream name pattern defined in
	 * auditlog-store.conf file
	 *
	 * @param domainUid the domain uid for the auditlog datastream to remove
	 * 
	 * @throws AuditLogCreationException
	 */
	@POST
	@Path("_remove_auditlog")
	public void removeAuditLogBackingStore(String domainUid);

	/*
	 * Checks the existence of an auditlog datastream
	 *
	 * @param domainUid the domain uid for the auditlog datastream to check
	 * 
	 * @return true if exists, false otherwise
	 * 
	 */
	@GET
	@Path("_has_auditlog")
	public boolean hasAuditLogBackingStore(String domainUid);

	/*
	 * Creates a datastream using datastream full name
	 *
	 * @param dataStreamFullName the full name of the datastream to create
	 * 
	 * @throws IOException
	 */
	@POST
	@Path("_create_datastream")
	public void createDataStream(String dataStreamFullName) throws IOException;

	/*
	 * Checks if a datastream exists
	 *
	 * @param dataStreamFullName the full name of the datastream to check
	 * 
	 * @throws IOException
	 */
	@GET
	@Path("_has_datastream")
	public boolean hasDataStream(String dataStreamFullName) throws IOException;

	/*
	 * Updates ILM policy retention duration
	 */
	@POST
	@Path("set_rentention")
	public void updateILMPolicyRetentionDuration(int duration) throws AuditLogILMPolicyException;

	/*
	 * Gets ILM policy retention duration (in days)
	 */
	@GET
	@Path("get_rentention")
	public String getRetentionDuration() throws AuditLogILMPolicyException;

	@GET
	@Path("remove_datastream")
	public void removeDataStream(String dataStreamFullName) throws AuditLogRemovalException;
}
