/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.core.backup.store.kafka.index;

import java.util.List;
import java.util.Properties;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicLong;
import java.util.function.BiPredicate;
import java.util.function.Consumer;

import org.apache.kafka.common.serialization.Serde;
import org.apache.kafka.common.utils.Bytes;
import org.apache.kafka.streams.KafkaStreams;
import org.apache.kafka.streams.KafkaStreams.State;
import org.apache.kafka.streams.KafkaStreams.StateListener;
import org.apache.kafka.streams.KeyValue;
import org.apache.kafka.streams.StreamsBuilder;
import org.apache.kafka.streams.StreamsConfig;
import org.apache.kafka.streams.kstream.Consumed;
import org.apache.kafka.streams.kstream.KStream;
import org.apache.kafka.streams.kstream.KTable;
import org.apache.kafka.streams.kstream.Materialized;
import org.apache.kafka.streams.state.KeyValueStore;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.util.concurrent.RateLimiter;

import net.bluemind.core.backup.continuous.index.IIndexBuilder;
import net.bluemind.core.backup.store.kafka.config.KafkaStoreConfig;

public class KTableIndexBuilder<K, V> implements IIndexBuilder<K, V> {

	private static final Logger logger = LoggerFactory.getLogger(KTableIndexBuilder.class);

	private KStream<K, V> stream;
	private final Properties props;
	private final StreamsBuilder builder;
	private final String idxName;

	private record BuilderAndStream<K, V>(StreamsBuilder builder, KStream<K, V> stream) {

	}

	public static <K, V> KTableIndexBuilder<K, V> indexBuilder(String bootstrap, List<String> topics, String indexName,
			Serde<K> keySerdes, Serde<V> valueSerdes) {
		ClassLoader savedCl = Thread.currentThread().getContextClassLoader();
		Thread.currentThread().setContextClassLoader(null);
		try {
			return new KTableIndexBuilder<>(bootstrap, topics, indexName, keySerdes, valueSerdes);
		} finally {
			Thread.currentThread().setContextClassLoader(savedCl);
		}

	}

	private static <K, V> BuilderAndStream<K, V> builder(List<String> topics, Serde<K> keySerdes,
			Serde<V> valueSerdes) {
		StreamsBuilder sb = new StreamsBuilder();
		KStream<K, V> stream = sb.<K, V>stream(topics, Consumed.with(keySerdes, valueSerdes));
		return new BuilderAndStream<>(sb, stream);

	}

	static final short REPL_FACTOR = (short) KafkaStoreConfig.get().getInt("kafka.topic.replicationFactor");

	private static Properties props(String bootstrap, String indexName) {

		Properties props = new Properties();
		props.put(StreamsConfig.APPLICATION_ID_CONFIG, "clone-idx-" + indexName + "-" + UUID.randomUUID().toString());
		props.put(StreamsConfig.BOOTSTRAP_SERVERS_CONFIG, bootstrap);
		props.put(StreamsConfig.REPLICATION_FACTOR_CONFIG, Short.toString(REPL_FACTOR));
		props.put(StreamsConfig.COMMIT_INTERVAL_MS_CONFIG, 5000);
		props.put(StreamsConfig.NUM_STREAM_THREADS_CONFIG, Math.max(2 * Runtime.getRuntime().availableProcessors(), 4));
		props.put(StreamsConfig.TOPOLOGY_OPTIMIZATION_CONFIG, StreamsConfig.OPTIMIZE);
		props.put(StreamsConfig.STATE_DIR_CONFIG, "/var/cache/bm-core/kafka-streams");
		return props;
	}

	public KTableIndexBuilder(String bootstrap, List<String> topics, String indexName, Serde<K> keySerdes,
			Serde<V> valueSerdes) {
		this(props(bootstrap, indexName), indexName, builder(topics, keySerdes, valueSerdes));
	}

	private KTableIndexBuilder(Properties props, String name, BuilderAndStream<K, V> builder) {
		this.props = props;
		this.builder = builder.builder;
		this.stream = builder.stream;
		this.idxName = name;
	}

	@Override
	public IIndexBuilder<K, V> filter(BiPredicate<K, V> pref) {
		this.stream = stream.filter(pref::test);
		return this;
	}

	@Override
	public <K2, V2> IIndexBuilder<K2, V2> map(Mapper<K, V, K2, V2> mapper) {
		KStream<K2, V2> restream = stream.map((k, v) -> {
			ReMapped<K2, V2> mapped = mapper.map(k, v);
			return new KeyValue<>(mapped.key(), mapped.value());
		});
		return new KTableIndexBuilder<>(props, idxName, new BuilderAndStream<>(builder, restream));
	}

	@Override
	public <K2, V2> IIndexBuilder<K2, V2> flatMap(ListMapper<K, V, K2, V2> mapper) {
		KStream<K2, V2> restream = stream.flatMap((k, v) -> {
			List<ReMapped<K2, V2>> asList = mapper.flatMap(k, v);
			return asList.stream().map(rm -> new KeyValue<K2, V2>(rm.key(), rm.value())).toList();
		});
		return new KTableIndexBuilder<>(props, idxName, new BuilderAndStream<>(builder, restream));
	}

	public static record TableAndStreams<K, V>(KTable<K, V> table, KafkaStreams streams, Listener progress) {

		public long globalLag() {
			return progress.currentLag();
		}

	}

	private static class Listener implements StateListener {

		private final KafkaStreams stream;
		private final RateLimiter logLimiter;
		private State state;
		private Consumer<Long> forLog;

		public Listener(KafkaStreams streams, Consumer<Long> forLog) {
			this.stream = streams;
			this.logLimiter = RateLimiter.create(1.0 / 5);
			this.forLog = forLog;
		}

		@Override
		public void onChange(State newState, State oldState) {
			this.state = newState;
		}

		public long currentLag() {
			if (state == State.RUNNING) {
				return computeLag();
			} else {
				return Long.MAX_VALUE;
			}
		}

		private long computeLag() {
			AtomicLong totalLag = new AtomicLong(0);
			stream.metadataForLocalThreads().forEach(tm -> {
				for (var at : tm.activeTasks()) {
					var commits = at.committedOffsets();
					var ends = at.endOffsets();
					commits.forEach((tp, commit) -> {
						if (ends.containsKey(tp)) {
							long endOfset = ends.get(tp);
							totalLag.addAndGet(Math.max(0, endOfset - commit));
						}
					});
				}
			});
			if (logLimiter.tryAcquire()) {
				forLog.accept(totalLag.get());
			}
			return totalLag.get();
		}

	}

	public TableAndStreams<K, V> build(Serde<K> key, Serde<V> value) {
		ClassLoader savedCl = Thread.currentThread().getContextClassLoader();
		Thread.currentThread().setContextClassLoader(null);
		try {
			Materialized<K, V, KeyValueStore<Bytes, byte[]>> target = Materialized
					.<K, V, KeyValueStore<Bytes, byte[]>>as(idxName).withKeySerde(key).withValueSerde(value)
					.withLoggingDisabled();
			KTable<K, V> ktable = stream.toTable(target);
			KafkaStreams streams = new KafkaStreams(builder.build(), props);
			Listener l = new Listener(streams, lag -> logger.info("[{} index] lag is {}", idxName, lag));
			streams.setStateListener(l);
			streams.start();
			return new TableAndStreams<>(ktable, streams, l);
		} finally {
			Thread.currentThread().setContextClassLoader(savedCl);
		}
	}

}
