package net.bluemind.core.backup.store.kafka;

import java.util.Map;
import java.util.Properties;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executor;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.ProducerConfig;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.clients.producer.RecordMetadata;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.base.MoreObjects;
import com.typesafe.config.Config;

import net.bluemind.core.backup.continuous.store.TopicPublisher;
import net.bluemind.core.backup.store.kafka.config.KafkaStoreConfig;
import net.bluemind.lifecycle.helper.SoftReset;

public class KafkaTopicPublisher implements TopicPublisher {

	private static final Logger logger = LoggerFactory.getLogger(KafkaTopicPublisher.class);

	private final String bootstrapServer;
	private final String physicalTopic;
	private final KafkaProducer<byte[], byte[]> producer;

	static final Map<String, KafkaProducer<byte[], byte[]>> perPhyTopicProd = perTopicProducers();

	private static Map<String, KafkaProducer<byte[], byte[]>> perTopicProducers() {
		Map<String, KafkaProducer<byte[], byte[]>> prods = new ConcurrentHashMap<>();
		SoftReset.register(() -> perPhyTopicProd.values().removeIf(prod -> {
			prod.close();
			logger.info("{} closed.", prod);
			return true;
		}));
		return prods;
	}

	public KafkaTopicPublisher(String bootstrapServer, String physicalTopic) {
		this.bootstrapServer = bootstrapServer;
		this.physicalTopic = physicalTopic;
		this.producer = perPhyTopicProd.computeIfAbsent(physicalTopic, s -> createKafkaProducer());
	}

	private static final Executor IN_ORDER = Executors.newSingleThreadExecutor();

	@Override
	public CompletableFuture<Void> store(String partitionToken, byte[] key, byte[] data) {
		CompletableFuture<Void> comp = new CompletableFuture<>();
		int partition = Math.abs(partitionToken.hashCode() % KafkaTopicStore.PARTITION_COUNT);
		ProducerRecord<byte[], byte[]> rec = new ProducerRecord<>(physicalTopic, partition, key, data);
		// send may block...
		// https://github.com/micronaut-projects/micronaut-kafka/issues/480
		IN_ORDER.execute(() -> {
			try {
				logger.trace("submit {}", rec);
				producer.send(rec, (RecordMetadata metadata, Exception exception) -> {
					if (exception != null) {
						logger.error("Could not store {}byte(s) of data. Key: {}, ({}) CRITICAL FAILURE",
								data == null ? 0 : data.length, new String(key), exception.getMessage());
						comp.completeExceptionally(exception);
					} else {
						logger.debug("[{}] stored part: {}, meta: {}", physicalTopic, partition, metadata);
						comp.complete(null);
					}
				});
			} catch (Exception e) {
				logger.error("Producer error", e);
				comp.completeExceptionally(e);
			}
		});

		return comp;
	}

	@Override
	public long maximumSize() {
		return KafkaStoreConfig.get().getMemorySize("kafka.producer.maxRecordSize").toBytes();
	}

	private KafkaProducer<byte[], byte[]> createKafkaProducer() {
		Config conf = KafkaStoreConfig.get();
		Properties producerProps = new Properties();
		producerProps.setProperty(ProducerConfig.BOOTSTRAP_SERVERS_CONFIG, bootstrapServer);
		producerProps.setProperty(ProducerConfig.COMPRESSION_TYPE_CONFIG, KafkaTopicStore.COMPRESSION_TYPE);
		producerProps.setProperty(ProducerConfig.KEY_SERIALIZER_CLASS_CONFIG,
				"org.apache.kafka.common.serialization.ByteArraySerializer");
		producerProps.setProperty(ProducerConfig.VALUE_SERIALIZER_CLASS_CONFIG,
				"org.apache.kafka.common.serialization.ByteArraySerializer");
		// https://docs.confluent.io/platform/current/installation/configuration/producer-configs.html#enable-idempotence
		producerProps.setProperty(ProducerConfig.MAX_IN_FLIGHT_REQUESTS_PER_CONNECTION, Integer.toString(5));
		producerProps.setProperty(ProducerConfig.ENABLE_IDEMPOTENCE_CONFIG, "true");

		// tunables
		producerProps.setProperty(ProducerConfig.ACKS_CONFIG, conf.getString("kafka.producer.acks"));
		producerProps.setProperty(ProducerConfig.LINGER_MS_CONFIG,
				Long.toString(conf.getDuration("kafka.producer.linger", TimeUnit.MILLISECONDS)));
		producerProps.setProperty(ProducerConfig.BUFFER_MEMORY_CONFIG,
				Long.toString(conf.getMemorySize("kafka.producer.bufferMemory").toBytes()));
		producerProps.setProperty(ProducerConfig.BATCH_SIZE_CONFIG,
				Long.toString(conf.getMemorySize("kafka.producer.batchSize").toBytes()));
		producerProps.setProperty(ProducerConfig.MAX_REQUEST_SIZE_CONFIG,
				Long.toString(conf.getMemorySize("kafka.producer.maxRecordSize").toBytes()));

		// https://stackoverflow.com/questions/37363119/kafka-producer-org-apache-kafka-common-serialization-stringserializer-could-no#:~:text=instance%20like%20this-,Thread.currentThread().setContextClassLoader(null)%3B%0AProducer%3CString%2C%20String%3E%20producer%20%3D%20new%20KafkaProducer(props)%3B,-hope%20my%20answer
		ClassLoader savedCl = Thread.currentThread().getContextClassLoader();
		Thread.currentThread().setContextClassLoader(null);
		try {
			return new KafkaProducer<>(producerProps);
		} finally {
			Thread.currentThread().setContextClassLoader(savedCl);
		}
	}

	@Override
	public String toString() {
		return MoreObjects.toStringHelper("KafkaTopic").add("name", physicalTopic).add("prod", producer).toString();
	}
}
