/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.server.cql.store;

import java.sql.SQLException;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.datastax.oss.driver.api.core.CqlSession;

import net.bluemind.core.container.model.Item;
import net.bluemind.cql.persistence.CqlAbstractStore;
import net.bluemind.server.api.Server;
import net.bluemind.server.repository.IServerStore;

public class CqlServerStore extends CqlAbstractStore implements IServerStore {

	public CqlServerStore(CqlSession s) {
		super(s);
	}

	private static final EntityPopulator<Server> POP = (r, i, v) -> {
		v.ip = r.getString(i++);
		v.fqdn = r.getString(i++);
		v.name = r.getString(i++);
		v.tags = List.copyOf(r.getSet(i++, String.class));
		return i;
	};
	private static final EntityPopulator<ServerWithId> POP_ID = (r, i, v) -> {
		v.ip = r.getString(i++);
		v.fqdn = r.getString(i++);
		v.name = r.getString(i++);
		v.tags = List.copyOf(r.getSet(i++, String.class));
		v.itemId = r.getLong(i++);
		return i;
	};

	@Override
	public void create(Item item, Server value) throws SQLException {
		store(item, value);
	}

	private void store(Item item, Server value) {
		voidCql("""
				INSERT INTO t_server
				(ip, fqdn, name, tags, item_id)
				VALUES
				(?, ?, ?, ?, ?)
				""", value.ip, value.fqdn, value.name, Set.copyOf(value.tags), item.id);
	}

	@Override
	public void update(Item item, Server value) throws SQLException {
		store(item, value);
	}

	@Override
	public void delete(Item item) throws SQLException {
		voidCql("DELETE FROM t_server WHERE item_id=?", item.id);
	}

	@Override
	public Server get(Item item) throws SQLException {
		return unique("SELECT ip, fqdn, name, tags FROM t_server WHERE item_id=?", r -> new Server(), POP, item.id);
	}

	@Override
	public void deleteAll() throws SQLException {
		throw new UnsupportedOperationException();
	}

	private static class ServerWithId extends Server { // NOSONAR
		long itemId;
	}

	@Override
	public List<Server> getMultiple(List<Item> items) throws SQLException {
		List<ServerWithId> srvIds = map("SELECT ip, fqdn, name, tags, item_id FROM t_server WHERE item_id IN ?",
				r -> new ServerWithId(), POP_ID, items.stream().map(i -> i.id).toList());
		Map<Long, ServerWithId> map = srvIds.stream().collect(Collectors.toMap(s -> s.itemId, s -> s, (s1, s2) -> s1));
		return items.stream().map(i -> (Server) map.get(i.id)).toList();
	}

	@Override
	public boolean exists(Item item) throws SQLException {
		Long found = unique("SELECT item_id FROM t_server WHERE item_id=?", r -> r.getLong(0), voidPop(), item.id);
		return found != null;
	}

}
