/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2024
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.tx.outbox.service;

import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.atomic.LongAdder;
import java.util.function.IntSupplier;
import java.util.function.Supplier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Lists;

import io.vertx.core.AbstractVerticle;
import io.vertx.core.Verticle;
import io.vertx.core.impl.ConcurrentHashSet;
import net.bluemind.configfile.core.CoreConfig;
import net.bluemind.core.backup.continuous.api.IBackupStore;
import net.bluemind.core.backup.continuous.api.IBackupStoreFactory;
import net.bluemind.core.backup.continuous.api.Providers;
import net.bluemind.core.container.model.BaseContainerDescriptor;
import net.bluemind.core.container.model.DataLocation;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.lib.vertx.IUniqueVerticleFactory;
import net.bluemind.lib.vertx.IVerticleFactory;
import net.bluemind.network.topology.Topology;
import net.bluemind.repository.provider.RepositoryProvider;
import net.bluemind.server.api.TagDescriptor;
import net.bluemind.system.api.SystemState;
import net.bluemind.system.state.StateContext;
import net.bluemind.tx.outbox.repository.ITxOutboxRepository;
import net.bluemind.tx.outbox.repository.ITxOutboxRepository.KafkaPayload;

public class TxOutboxBackupStoreFlusher extends AbstractVerticle {

	private static final Logger logger = LoggerFactory.getLogger(TxOutboxBackupStoreFlusher.class);

	public static final IntSupplier SLICE = () -> CoreConfig.get().getInt(CoreConfig.TxOutbox.SLICE);
	public static final IntSupplier MGET = () -> CoreConfig.get().getInt(CoreConfig.TxOutbox.MGET);

	private static final Set<SystemState> FLUSH_SAFE = EnumSet.of(//
			SystemState.CORE_STATE_STARTING, //
			SystemState.CORE_STATE_RUNNING, //
			SystemState.CORE_STATE_DEMOTED//
	);

	private final Supplier<ITxOutboxRepository> repository;
	private final Set<Long> inFlight;
	private final LongAdder flushed;
	private final String logName;

	public TxOutboxBackupStoreFlusher(String logName, Supplier<ITxOutboxRepository> repository) {
		this.repository = repository;
		this.flushed = new LongAdder();
		this.inFlight = new ConcurrentHashSet<>(4 * SLICE.getAsInt());
		this.logName = logName;
		reportRate();
	}

	private void reportRate() {
		Thread.ofVirtual().name("outbox-" + logName).start(() -> {
			while (true) {
				long cur = flushed.sum();
				try {
					Thread.sleep(10000);
				} catch (InterruptedException e) {
					Thread.currentThread().interrupt();
					break;
				}
				long tenSecLater = flushed.sum();
				long diff = tenSecLater - cur;
				long perSec = diff / 10;

				if (perSec > 0) {
					logger.info("Flushed {} in 10sec ({} since start), rate: {}/s", diff, tenSecLater, perSec);
				}
			}
		});
	}

	@Override
	public void start() throws Exception {
		scheduleFlush(500);
	}

	private long scheduleFlush(long ms) {
		Thread.ofVirtual().name("outbox-flusher-" + logName).start(() -> {
			if (ms > 0) {
				try {
					Thread.sleep(ms);
				} catch (InterruptedException e) {
					Thread.currentThread().interrupt();
					return;
				}
			}
			flushSlice();
		});
		return ms;
	}

	private static record Result(long outboxId, Throwable failed) {
	}

	public long flushSlice() {
		IBackupStoreFactory bsf = Providers.get();

		if (bsf.isPaused() || !FLUSH_SAFE.contains(StateContext.getState()) || !bsf.leadership().isLeader()) {
			if (logger.isDebugEnabled()) {
				logger.debug("Scheduling for later, bsf.paused: {}, state: {}, leader: {}", bsf.isPaused(),
						StateContext.getState(), bsf.leadership().isLeader());
			}
			return scheduleFlush(500);
		}

		SystemState state = StateContext.getState();
		if (!FLUSH_SAFE.contains(state)) {
			return scheduleFlush(500);
		}

		if (!bsf.leadership().isLeader()) {
			return scheduleFlush(500);
		}
		if (inFlight.size() > 2 * SLICE.getAsInt()) {
			return scheduleFlush(5);
		}

		try {
			ITxOutboxRepository outboxRepo = repository.get();
			List<Long> slice = outboxRepo.kafkaPending(SLICE.getAsInt(), List.copyOf(inFlight));
			slice.removeIf(inFlight::contains);
			for (List<Long> chunk : Lists.partition(slice, MGET.getAsInt())) {
				List<CompletableFuture<Result>> proms = new ArrayList<>(chunk.size());
				List<KafkaPayload> payloads = outboxRepo.mget(chunk);
				for (KafkaPayload payload : payloads) {
					BaseContainerDescriptor bcd = BaseContainerDescriptor.create("dummy_uid", "dummy_name",
							payload.partKey(), "type", payload.domainUid(), false);
					IBackupStore<Object> store = bsf.forContainer(bcd);
					if (logger.isTraceEnabled()) {
						logger.trace("Flush d:{},o:{} to {}", payload.domainUid(), payload.partKey(), store);
					}
					byte[] content = payload.value();
					if (content.length == 0) {
						content = null;
					}
					proms.add(store.storeRaw(payload.partKey(), payload.key(), content)
							.thenApply(v -> new Result(payload.outboxId(), null)).exceptionally(t -> {
								logger.error("Kafka definitive reject of {}: {}", payload.outboxId(), t.getMessage());
								return new Result(0L, t);
							}));
				}
				inFlight.addAll(chunk);
				CompletableFuture<Void> flushProm = CompletableFuture
						.allOf(proms.toArray(new CompletableFuture<?>[proms.size()]));
				flushProm.thenAccept(done -> {
					List<Long> ok = proms.stream().map(t -> t.join()).filter(r -> r.failed == null).map(r -> r.outboxId)
							.toList();
					chunkCompleted(outboxRepo, ok, chunk);
				}).exceptionally(t -> {
					logger.error(t.getMessage(), t);
					return null;
				});
			}
		} catch (Exception e) {
			logger.error("Kafka outbox transient failure", e);
		}

		return scheduleFlush(inFlight.isEmpty() ? 500 : 0);
	}

	private void chunkCompleted(ITxOutboxRepository outboxRepo, List<Long> pushed, List<Long> attempted) {
		try {
			outboxRepo.deleteOffsets(pushed);
			attempted.forEach(inFlight::remove);
			flushed.add(pushed.size());
			if (!pushed.isEmpty()) {
				outboxRepo.seqHolder().flushed.updateAndGet(cur -> Math.max(cur, pushed.getLast()));
			}
		} catch (Exception t) {
			logger.error(t.getMessage(), t);
		}
	}

	public static class RegDirectory implements IVerticleFactory, IUniqueVerticleFactory {
		@Override
		public boolean isWorker() {
			return true;
		}

		@Override
		public Verticle newInstance() {
			return new TxOutboxBackupStoreFlusher("BJ", () -> {
				ServerSideServiceProvider sp = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM);
				return RepositoryProvider.instance(ITxOutboxRepository.class, sp.getContext(),
						DataLocation.directory());
			});
		}
	}

	public static class RegBjData implements IVerticleFactory, IUniqueVerticleFactory {
		@Override
		public boolean isWorker() {
			return true;
		}

		@Override
		public Verticle newInstance() {
			return new TxOutboxBackupStoreFlusher("BJ-DATA", () -> {
				ServerSideServiceProvider sp = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM);
				return RepositoryProvider.instance(ITxOutboxRepository.class, sp.getContext(),
						mboxLocations().getFirst());
			});
		}

		private List<DataLocation> mboxLocations() {
			return Topology.getIfAvailable()
					.map(tp -> tp.all(TagDescriptor.bm_pgsql_data.getTag(), TagDescriptor.mail_imap.getTag())//
							.stream().distinct().map(iv -> DataLocation.of(iv.uid)).toList())
					.orElseGet(Collections::emptyList);
		}
	}

}
