/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2022
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.notes.impl.pg;

import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.google.common.collect.Lists;

import net.bluemind.core.container.api.ContainerQuery;
import net.bluemind.core.container.api.IContainers;
import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.BmContext;
import net.bluemind.core.rest.ServerSideServiceProvider;
import net.bluemind.core.task.service.IServerTaskMonitor;
import net.bluemind.dataprotect.api.DPError;
import net.bluemind.dataprotect.api.DPError.DPKind;
import net.bluemind.dataprotect.api.DataProtectGeneration;
import net.bluemind.dataprotect.common.restore.IMonitoredRestoreRestorableItem;
import net.bluemind.dataprotect.service.BackupDataProvider;
import net.bluemind.notes.api.INote;
import net.bluemind.notes.api.INoteUids;
import net.bluemind.notes.api.VNote;
import net.bluemind.notes.api.VNoteChanges;

public class RestoreNotesTaskPg {
	private static final Logger logger = LoggerFactory.getLogger(RestoreNotesTaskPg.class);

	private final DataProtectGeneration backup;
	private final IMonitoredRestoreRestorableItem restorableItem;

	/**
	 * @deprecated REMOVE THIS IN BlueMind 6+
	 */
	@Deprecated
	public RestoreNotesTaskPg(DataProtectGeneration backup, IMonitoredRestoreRestorableItem item) {
		this.backup = backup;
		this.restorableItem = item;
	}

	public void run() throws Exception {
		try (BackupDataProvider bdp = new BackupDataProvider(null, SecurityContext.SYSTEM,
				restorableItem.getMonitor())) {
			BmContext back = bdp.createContextWithData(backup, restorableItem.item());
			BmContext live = ServerSideServiceProvider.getProvider(SecurityContext.SYSTEM).getContext();

			IContainers liveContApi = live.provider().instance(IContainers.class);
			List<ContainerDescriptor> liveLists = liveContApi
					.all(ContainerQuery.ownerAndType(restorableItem.liveEntryUid(), INoteUids.TYPE));

			IContainers backContApi = back.provider().instance(IContainers.class);
			List<ContainerDescriptor> backLists = backContApi
					.all(ContainerQuery.ownerAndType(restorableItem.entryUid(), INoteUids.TYPE));

			restorableItem.monitorLog("Restoring '{}' note(s) from uid={}", backLists.size(),
					restorableItem.entryUid());

			logger.info("Backup contains {} note(s)", backLists.size());
			for (ContainerDescriptor backList : backLists) {
				restore(back, live, backList, liveLists, restorableItem.monitorSubWork(1));
			}
		} catch (Exception e) {
			logger.error("Error while restoring notes", e);
			restorableItem.errors().add(DPError.restore(e.getMessage(), restorableItem.entryUid(), DPKind.NOTES));
		} finally {
			restorableItem.endTask();
		}
	}

	private void restore(BmContext back, BmContext live, ContainerDescriptor backList,
			List<ContainerDescriptor> liveLists, IServerTaskMonitor monitor) {
		IContainers liveContApi = live.provider().instance(IContainers.class);
		INote backupApi = back.provider().instance(INote.class, backList.uid);

		List<String> allUids = backupApi.allUids();
		restorableItem.monitorLog("Restoring {} notes for {} [uid={}]", allUids.size(), backList.name, backList.uid);

		String listUid = mapListUid(backList.uid);

		if (liveLists.stream().anyMatch(c -> c.uid.equals(listUid))) {
			INote liveABApi = live.provider().instance(INote.class, listUid);
			liveABApi.reset();
			monitor.progress(1, "reset done");
		} else {
			backList.owner = restorableItem.liveEntryUid();
			liveContApi.create(listUid, backList);
			monitor.progress(1, "notes recreated");
		}

		INote liveApi = live.provider().instance(INote.class, listUid);

		for (List<String> batch : Lists.partition(backupApi.allUids(), 1000)) {
			List<ItemValue<VNote>> notes = backupApi.multipleGet(batch);
			VNoteChanges changes = VNoteChanges.create(notes.stream()
					.map(e -> VNoteChanges.ItemAdd.create(e.uid, e.value, false)).collect(Collectors.toList()),
					Collections.emptyList(), Collections.emptyList());
			liveApi.updates(changes);
			monitor.progress(batch.size(), null);
		}
	}

	private String mapListUid(String uid) {
		if (!restorableItem.entryUid().equals(restorableItem.liveEntryUid())
				&& uid.endsWith(String.format("_%s", restorableItem.entryUid()))) {
			return String.format("%s%s", uid.substring(0, uid.length() - restorableItem.entryUid().length()),
					restorableItem.liveEntryUid());
		}

		return uid;
	}
}
