/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2025
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.core.backup.continuous.restore.domains.crud;

import com.fasterxml.jackson.core.type.TypeReference;

import net.bluemind.core.backup.continuous.model.RecordKey;
import net.bluemind.core.backup.continuous.model.RecordKey.Operation;
import net.bluemind.core.backup.continuous.restore.domains.RestoreDomainType;
import net.bluemind.core.backup.continuous.restore.domains.RestoreLogger;
import net.bluemind.core.container.model.ItemValue;
import net.bluemind.core.rest.IServiceProvider;
import net.bluemind.core.utils.JsonUtils;
import net.bluemind.core.utils.JsonUtils.ValueReader;
import net.bluemind.domain.api.Domain;
import net.bluemind.system.api.Credential;
import net.bluemind.system.api.IInternalCredentials;
import net.bluemind.system.api.KafkaCredentialEntry;

public class RestoreUserCredentials implements RestoreDomainType {

	private static final ValueReader<ItemValue<KafkaCredentialEntry>> reader = JsonUtils
			.reader(new TypeReference<ItemValue<KafkaCredentialEntry>>() {
			});
	private final IServiceProvider target;
	private RestoreLogger log;
	private ItemValue<Domain> domain;

	public RestoreUserCredentials(RestoreLogger log, ItemValue<Domain> domain, IServiceProvider target) {
		this.target = target;
		this.log = log;
		this.domain = domain;
	}

	@Override
	public String type() {
		return KafkaCredentialEntry.TYPE;
	}

	@Override
	public void restore(RecordKey key, String payload) {
		IInternalCredentials apiCredentials = target.instance(IInternalCredentials.class, domain.uid);

		switch (Operation.of(key)) {
		case Operation.CREATE: {
			log.create(type(), key);
			ItemValue<KafkaCredentialEntry> itemvalue = reader.read(payload);
			Credential entry = itemvalue.value.credential();
			apiCredentials.restoreUserCredential(itemvalue.value.userUid(), entry);
			break;
		}
		case Operation.DELETE: {
			log.delete(type(), key);
			ItemValue<KafkaCredentialEntry> itemvalue = reader.read(payload);
			apiCredentials.deleteUserCredential(itemvalue.value.userUid(), itemvalue.value.credential().id);
			break;
		}
		default:
			IllegalArgumentException exception = new IllegalArgumentException("Unsupported operation: " + key);
			log.failure(payload, key, payload, exception);
			throw exception;
		}
	}
}