/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2016
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.dataprotect.calendar.impl.pg;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.ResourceBundle;
import java.util.UUID;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.calendar.api.ICalendarUids;
import net.bluemind.calendar.api.IVEvent;
import net.bluemind.core.api.fault.ServerFault;
import net.bluemind.core.container.api.ContainerQuery;
import net.bluemind.core.container.api.IContainers;
import net.bluemind.core.container.model.ContainerDescriptor;
import net.bluemind.core.context.SecurityContext;
import net.bluemind.core.rest.IServiceProvider;
import net.bluemind.core.rest.base.GenericStream;
import net.bluemind.core.sessions.Sessions;
import net.bluemind.dataprotect.api.DPError;
import net.bluemind.dataprotect.api.DataProtectGeneration;
import net.bluemind.dataprotect.common.email.SendIcs;
import net.bluemind.dataprotect.common.restore.IMonitoredRestoreRestorableItem;
import net.bluemind.dataprotect.service.BackupDataProvider;
import net.bluemind.dataprotect.service.action.EmailData;
import net.bluemind.dataprotect.service.action.IRestoreActionData;
import net.bluemind.dataprotect.service.action.RestoreActionExecutor;

public class SendUserCalendarsICSTaskPg {
	private static final Logger logger = LoggerFactory.getLogger(SendUserCalendarsICSTaskPg.class);

	private final DataProtectGeneration backup;
	private IMonitoredRestoreRestorableItem restorableItem;
	private final RestoreActionExecutor<EmailData> executor;
	private ResourceBundle bundle;

	/**
	 * @deprecated REMOVE THIS IN BlueMind 6+
	 */
	@Deprecated
	@SuppressWarnings("unchecked")
	public SendUserCalendarsICSTaskPg(DataProtectGeneration backup, IMonitoredRestoreRestorableItem item,
			RestoreActionExecutor<? extends IRestoreActionData> executor, ResourceBundle bundle) {
		this.backup = backup;
		this.bundle = bundle;
		this.restorableItem = item;
		this.executor = (RestoreActionExecutor<EmailData>) executor;
	}

	public static final SecurityContext as(String uid, String domainContainerUid) throws ServerFault {
		SecurityContext userContext = new SecurityContext(UUID.randomUUID().toString(), uid, Arrays.<String>asList(),
				Arrays.<String>asList(), Collections.emptyMap(), domainContainerUid, "en",
				"SendUserCalendarsICSTask.as");
		Sessions.get().put(userContext.getSessionId(), userContext);
		return userContext;
	}

	public void run() throws Exception {
		logger.info("Starting export for uid {}", restorableItem.entryUid());
		SendIcs sendEmail = new SendIcs(restorableItem, executor);

		SecurityContext backUserContext = as(restorableItem.entryUid(), restorableItem.domain());
		try (BackupDataProvider bdp = new BackupDataProvider(null, backUserContext, restorableItem.getMonitor())) {
			IServiceProvider back = bdp.createContextWithData(backup, restorableItem.item()).provider();

			IContainers containersService = back.instance(IContainers.class);
			ContainerQuery cq = ContainerQuery.ownerAndType(backUserContext.getSubject(), ICalendarUids.TYPE);
			List<ContainerDescriptor> cals = containersService.all(cq);

			Map<String, String> allIcs = HashMap.newHashMap(cals.size());
			for (ContainerDescriptor cal : cals) {
				IVEvent service = back.instance(IVEvent.class, cal.uid);
				allIcs.put(cal.name, GenericStream.streamToString(service.exportAll()));
			}

			restorableItem.monitorLog("Export '{}' events from {} [uid={}]", allIcs.size(),
					restorableItem.displayName(), restorableItem.entryUid());

			sendEmail.sendMessage(allIcs, bundle.getString("send.calendar.restore.message"),
					bundle.getString("send.calendar.restore.subject"));

		} catch (Exception e) {
			logger.error("Error while sending user calendars", e);
			restorableItem.errors()
					.add(DPError.export(e.getMessage(), restorableItem.entryUid(), DPError.DPKind.CALENDAR));
		} finally {
			restorableItem.endTask();
		}
	}
}
