//@vitest-environment happy-dom
import { mount } from "@vue/test-utils";
import { vi } from "vitest";
import { Bluetiful, useRegisterSystemAlertArea, BmSystemAlertArea } from "@bluemind/bluetiful";
import { useTimezoneChecker } from "../composables/useTimezoneChecker";
import { useCookieSession } from "../composables/useCookieSession";
import store from "@bluemind/store";

const timezoneMock = function (zone) {
    const DateTimeFormat = Intl.DateTimeFormat;
    vi.spyOn(global.Intl, "DateTimeFormat").mockImplementation(
        (locale, options) => new DateTimeFormat(locale, { ...options, timeZone: zone })
    );
};

vi.mock("@bluemind/store", () => ({
    default: {
        dispatch: (_, arg) => (arg.setting === "timezone" ? (store.state.settings.timezone = arg.value) : vi.fn()),
        state: {
            settings: { timezone: "America/Chicago", timezone_difference_reminder: "true" }
        }
    }
}));

const cookieMockValues = {};
vi.mock("../composables/useCookieSession", () => ({
    useCookieSession: (name, defaultValue) => {
        if (name === "userAlertTimezonePreferences" && !cookieMockValues[name]) {
            cookieMockValues[name] = {
                timezone: Intl.DateTimeFormat().resolvedOptions().timeZone,
                alertClosedByUser: false
            };
        } else if (defaultValue && !cookieMockValues[name]) {
            cookieMockValues[name] = defaultValue;
        }

        return {
            getValue: () =>
                cookieMockValues[name] || {
                    timezone: Intl.DateTimeFormat().resolvedOptions().timeZone,
                    alertClosedByUser: false
                },
            setValue: value => {
                cookieMockValues[name] = value;
            }
        };
    }
}));
describe("useTimezoneChecker", () => {
    afterEach(() => {
        vi.restoreAllMocks();
    });
    it("Shouldn't display an alert when timezone are the same", async () => {
        setTestEnv({
            systemeTimezone: "America/Chicago",
            cookie: { timezone: "America/Chicago", closedByUser: false },
            settings: { timezone: "America/Chicago", timezone_difference_reminder: "true" }
        });

        let spy;
        const wrapper = mount(
            {
                template: `<div><BmSystemAlertArea class="system-alert-area" /></div>`,
                setup() {
                    const service = useRegisterSystemAlertArea();
                    spy = vi.spyOn(service, "dispatchSystemAlert").mockImplementation(() => vi.fn());
                    const { checkTimezone } = useTimezoneChecker(service);
                    return { checkTimezone };
                },
                components: { BmSystemAlertArea }
            },
            {
                global: {
                    plugins: [Bluetiful]
                }
            }
        );

        wrapper.vm.checkTimezone();

        expect(spy).not.toHaveBeenCalled();
    });
    it("Should display an alert when timezone are different", async () => {
        setTestEnv({
            systemeTimezone: "Europe/Paris",
            cookie: { timezone: "Europe/Paris", closedByUser: false },
            settings: { timezone: "America/Chicago", timezone_difference_reminder: "true" }
        });

        let spy;
        const wrapper = mount(
            {
                template: `<div><BmSystemAlertArea class="system-alert-area" /></div>`,
                setup() {
                    const service = useRegisterSystemAlertArea();
                    spy = vi.spyOn(service, "dispatchSystemAlert").mockImplementation(() => vi.fn());
                    const { checkTimezone } = useTimezoneChecker(service);
                    return { checkTimezone };
                },
                components: { BmSystemAlertArea }
            },
            {
                global: {
                    plugins: [Bluetiful]
                }
            }
        );

        wrapper.vm.checkTimezone();

        expect(spy).toHaveBeenCalled();
    });
    it("Shouldn't display an alert if reminder is false ", async () => {
        setTestEnv({
            systemeTimezone: "Europe/Paris",
            cookie: { timezone: "Europe/Paris", closedByUser: false },
            settings: { timezone: "America/Chicago", timezone_difference_reminder: "false" }
        });

        let spy;
        const wrapper = mount(
            {
                template: `<div><BmSystemAlertArea class="system-alert-area" /></div>`,
                setup() {
                    const service = useRegisterSystemAlertArea();
                    spy = vi.spyOn(service, "dispatchSystemAlert").mockImplementation(() => vi.fn());
                    const { checkTimezone } = useTimezoneChecker(service);
                    return { checkTimezone };
                },
                components: { BmSystemAlertArea }
            },
            {
                global: {
                    plugins: [Bluetiful]
                }
            }
        );

        wrapper.vm.checkTimezone();

        expect(spy).not.toHaveBeenCalled();
    });
    it("Shouldn't display an alert after user closed it", async () => {
        setTestEnv({
            systemeTimezone: "Europe/Paris",
            cookie: { timezone: "Europe/Paris", closedByUser: true },
            settings: { timezone: "America/Chicago", timezone_difference_reminder: "true" }
        });
        let spy;
        const wrapper = mount(
            {
                template: `<div><BmSystemAlertArea class="system-alert-area" /></div>`,
                setup() {
                    const service = useRegisterSystemAlertArea();
                    spy = vi.spyOn(service, "dispatchSystemAlert").mockImplementation(() => vi.fn());
                    const { checkTimezone } = useTimezoneChecker(service);
                    return { checkTimezone };
                },
                components: { BmSystemAlertArea }
            },
            {
                global: {
                    plugins: [Bluetiful]
                }
            }
        );
        simulateUserCloseAlert();

        wrapper.vm.checkTimezone();

        expect(spy).not.toHaveBeenCalled();
    });
});

function simulateUserCloseAlert() {
    setTimezoneCookie(Intl.DateTimeFormat().resolvedOptions().timeZone, true);
}
function setTimezoneCookie(timezone, closeByUser) {
    useCookieSession("userAlertTimezonePreferences").setValue({
        timezone: timezone,
        alertClosedByUser: closeByUser
    });
}
function setTestEnv({ systemeTimezone, cookie, settings }) {
    timezoneMock(systemeTimezone);
    setTimezoneCookie(cookie.timezone, cookie.closedByUser);
    store.state.settings = settings;
}
