/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2017
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.backend.mail.replica.persistence;

import java.sql.Array;
import java.sql.SQLException;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import javax.sql.DataSource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import net.bluemind.backend.mail.api.FolderCounters;
import net.bluemind.backend.mail.replica.api.AppendTx;
import net.bluemind.backend.mail.replica.api.MailboxReplica;
import net.bluemind.backend.mail.repository.IMailboxReplicaStore;
import net.bluemind.core.container.model.Container;
import net.bluemind.core.container.model.Item;
import net.bluemind.core.container.persistence.AbstractItemValueStore;
import net.bluemind.core.container.persistence.StringCreator;

public class MailboxReplicaStore extends AbstractItemValueStore<MailboxReplica> implements IMailboxReplicaStore {

	private static final Logger logger = LoggerFactory.getLogger(MailboxReplicaStore.class);
	private static final Creator<MailboxReplica> MB_CREATOR = con -> new MailboxReplica();
	private final Container container;
	public final String partition;

	public MailboxReplicaStore(DataSource pool, Container container, String partition) {
		super(pool);
		this.container = container;
		this.partition = partition;
		logger.debug("Created for {}", this.partition);
	}

	@Override
	public void create(Item item, MailboxReplica value) throws SQLException {
		String query = "INSERT INTO t_mailbox_replica (" + MailboxReplicaColumns.COLUMNS.names()
				+ ", unique_id, container_id, item_id) VALUES (" + MailboxReplicaColumns.COLUMNS.values()
				+ ", ?, ?, ?)";
		insert(query, value, MailboxReplicaColumns.values(container, item));
	}

	@Override
	public void update(Item item, MailboxReplica value) throws SQLException {
		String query = "UPDATE t_mailbox_replica SET (" + MailboxReplicaColumns.COLUMNS_UPD.names() + ") = ("
				+ MailboxReplicaColumns.COLUMNS_UPD.values() + ")" + " WHERE container_id = ? AND item_id = ?";
		update(query, value, MailboxReplicaColumns.updateValues(container, item));
	}

	@Override
	public void delete(Item item) throws SQLException {
		delete("DELETE FROM t_mailbox_replica WHERE container_id = ? AND item_id = ?",
				new Object[] { container.id, item.id });
	}

	private static final String GET_QUERY = "SELECT " + MailboxReplicaColumns.COLUMNS.names()
			+ " FROM t_mailbox_replica WHERE container_id = ? AND item_id = ?";

	@Override
	public MailboxReplica get(Item item) throws SQLException {
		return unique(GET_QUERY, MB_CREATOR, MailboxReplicaColumns.populator(), new Object[] { container.id, item.id });
	}

	@Override
	public void deleteAll() throws SQLException {
		delete("DELETE FROM t_mailbox_replica WHERE container_id = ?", new Object[] { container.id });
	}

	@Override
	public String byName(String name) throws SQLException {
		String query = "SELECT unique_id FROM t_mailbox_replica WHERE container_id = ? AND name = ?";
		return unique(query, StringCreator.FIRST, Collections.emptyList(), new Object[] { container.id, name });
	}

	private static final String APPEND_QUERY = "UPDATE t_mailbox_replica SET last_uid = last_uid+?, "
			+ "last_append_date=now() WHERE container_id = ? AND item_id = ? RETURNING last_uid, last_append_date";

	@Override
	public AppendTx prepareAppend(int count, long mboxReplicaId) throws SQLException {
		return unique(APPEND_QUERY, con -> new AppendTx(), (rs, idx, tx) -> {
			tx.imapUid = rs.getLong(idx++);
			tx.internalStamp = rs.getTimestamp(idx++).getTime();
			return idx;
		}, new Object[] { count, container.id, mboxReplicaId });
	}

	private static List<Long> listOfLong(Array array) throws SQLException {
		if (array == null) {
			return Collections.emptyList();
		}
		return Arrays.asList((Long[]) array.getArray());
	}

	@Override
	public List<Long> childrensOf(String folderUid) throws SQLException {
		String query = """
				WITH RECURSIVE folder_tree AS (
				  SELECT tmr.item_id, tmr.unique_id, tmr.container_id
				  FROM t_mailbox_replica tmr
				  WHERE tmr.container_id = ? AND tmr.unique_id = ?

				  UNION ALL

				  SELECT tmr.item_id, tmr.unique_id, tmr.container_id
				  FROM t_mailbox_replica tmr
				  JOIN folder_tree ft ON tmr.parent_uid = ft.unique_id AND tmr.container_id = ft.container_id
				)
				SELECT array_agg(folder_tree.item_id) FROM folder_tree;
				""";
		return unique(query, rs -> listOfLong(rs.getArray(1)), Collections.emptyList(),
				new Object[] { container.id, folderUid });
	}

	@Override
	public boolean exists(Item item) throws SQLException {
		String q = "SELECT 1 FROM t_mailbox_replica WHERE container_id = ? AND item_id = ?";
		return unique(q, rs -> true, (rs, index, v) -> index, new Object[] { container.id, item.id }) != null;
	}

	@Override
	public List<FolderCounters> counters(List<Long> folderItemIds) throws SQLException {
		String q = """
				WITH cuids AS (
				    SELECT
				        'mbox_records_' || unique_id AS uid,
				        item_id
				    FROM t_mailbox_replica
				    WHERE container_id=? AND item_id = ANY(?)
				),
				conts AS (
				    SELECT
				        c.id,
				        cuids.item_id
				    FROM cuids
				    INNER JOIN t_container c ON cuids.uid = c.uid
				),
				unseens AS (
				    SELECT
				        conts.item_id,
				        cnt.unseen_visible,
				        cnt.total_visible,
				        cnt.total,
				        seq.seq
				    FROM conts
				    INNER JOIN v_container_item_counter cnt ON conts.id = cnt.container_id
				    INNER JOIN t_container_sequence seq ON seq.container_id = conts.id
				)
				SELECT item_id, unseen_visible, total_visible, total, seq
				FROM unseens;""";
		return select(q, rs -> new FolderCounters(), (rs, idx, fc) -> {
			fc.itemId = rs.getLong(idx++);
			fc.unseenVisible = rs.getInt(idx++);
			fc.totalVisible = rs.getInt(idx++);
			fc.total = rs.getInt(idx++);
			fc.recordsVersion = rs.getLong(idx++);
			return idx;
		}, new Object[] { container.id, folderItemIds.toArray(Long[]::new) });
	}
}
