/* BEGIN LICENSE
 * Copyright © Blue Mind SAS, 2012-2018
 *
 * This file is part of BlueMind. BlueMind is a messaging and collaborative
 * solution.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of either the GNU Affero General Public License as
 * published by the Free Software Foundation (version 3 of the License).
 *
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 *
 * See LICENSE.txt
 * END LICENSE
 */
package net.bluemind.directory.hollow.datamodel;

import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.netflix.hollow.api.objects.HollowSet;
import com.netflix.hollow.core.write.objectmapper.HollowPrimaryKey;

import net.bluemind.directory.hollow.datamodel.consumer.HString;

@HollowPrimaryKey(fields = { "domainName" })
public class OfflineAddressBook {

	public String domainName;
	public Set<String> domainAliases;

	/**
	 * Display name of the address list. Can change between generation versions of
	 * the same address list.
	 */
	public String name;

	/**
	 * The AddressList-X500-DN of the address list container object. Can change
	 * between generation versions of the same address list. MUST contain Teletex
	 * characters only, as specified by the non-space-teletex rule in section 2.1.
	 */
	public String distinguishedName;

	/**
	 * The sequence number of the OAB. This number increases by one between
	 * generation versions of the same address list.
	 */
	public int sequence;

	/**
	 *
	 * A string formatted GUID that represents the address list container object.
	 * This value never changes between generation versions of the same address
	 * list. This value is formatted as "xxxxxxxx-xxxx-xxxx-xxxx-xxxxxxxxxxxx". MUST
	 * contain Teletex characters only, as specified by the non-space-teletex rule
	 * in section 2.1
	 */
	public String containerGuid;

	/**
	 * optional.
	 *
	 * DN for the root departmental group in the department hierarchy for the
	 * organization. The DN (3) can change between generation versions of the same
	 * address list. MUST contain Teletex characters only, as specified by the
	 * non-space-teletex rule in section 2.1.
	 *
	 */
	public String hierarchicalRootDepartment;

	private static <V, R> R ifNotNul(V source, Function<V, R> map) {
		return Optional.ofNullable(source).map(map).orElse(null);
	}

	private static String getValue(HString source) {
		return ifNotNul(source, HString::getValue);
	}

	private static <T1, T2> Set<T2> getValue(HollowSet<T1> source, Function<T1, T2> map) {
		return ifNotNul(source, s -> s.stream().map(map).collect(Collectors.toSet()));
	}

	public OfflineAddressBook() {
	}

	public OfflineAddressBook(net.bluemind.directory.hollow.datamodel.consumer.OfflineAddressBook source) {
		this(getValue(source.getDomainName()), //
				getValue(source.getDomainAliases(), HString::getValue), //
				getValue(source.getName()), //
				getValue(source.getDistinguishedName()), //
				source.getSequence(), //
				getValue(source.getContainerGuid()), //
				getValue(source.getHierarchicalRootDepartment()) //
		);
	}

	public OfflineAddressBook(String domainName, //
			Set<String> domainAliases, //
			String name, //
			String distinguishedName, //
			int sequence, //
			String containerGuid, //
			String hierarchicalRootDepartment //
	) {
		super();
		this.domainName = domainName;
		this.domainAliases = domainAliases;
		this.name = name;
		this.distinguishedName = distinguishedName;
		this.sequence = sequence;
		this.containerGuid = containerGuid;
		this.hierarchicalRootDepartment = hierarchicalRootDepartment;
	}

	public String getDomainName() {
		return domainName;
	}

	public Set<String> getDomainAliases() {
		return domainAliases;
	}

	public String getName() {
		return name;
	}

	public String getDistinguishedName() {
		return distinguishedName;
	}

	public int getSequence() {
		return sequence;
	}

	public String getContainerGuid() {
		return containerGuid;
	}

	public String getHierarchicalRootDepartment() {
		return hierarchicalRootDepartment;
	}

}
