/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2023
  *
  * This file is part of BlueMind. BlueMind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License).
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.cli.security;

import java.util.HashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import net.bluemind.cli.cmd.api.CliContext;
import net.bluemind.cli.cmd.api.ICmdLet;
import net.bluemind.cli.cmd.api.ICmdLetRegistration;
import net.bluemind.system.api.ISystemConfiguration;
import net.bluemind.system.api.SysConfKeys;
import net.bluemind.system.api.SystemConf;
import picocli.CommandLine.Command;
import picocli.CommandLine.IExitCodeGenerator;
import picocli.CommandLine.Option;

@Command(name = "host-whitelist", description = "Adds or removes a host(FQDN) from the whitelist")
public class HostWhitelistCommand implements ICmdLet, Runnable, IExitCodeGenerator {

	private int exitCode = 0;

	public static class Reg implements ICmdLetRegistration {

		@Override
		public Optional<String> group() {
			return Optional.of("security");
		}

		@Override
		public Class<? extends ICmdLet> commandClass() {
			return HostWhitelistCommand.class;
		}
	}

	private CliContext ctx;

	@Option(required = false, names = { "--list" }, description = "List the whitelisted hosts")
	public boolean list;

	@Option(required = false, names = { "--remove" }, description = "Remove the host")
	public boolean remove;

	@Option(required = false, names = { "--fqdn" }, description = "FQDN")
	public String fqdn;

	@Override
	public void run() {
		ISystemConfiguration service = ctx.adminApi().instance(ISystemConfiguration.class);
		Set<String> whitelist = new HashSet<>(service.getValues().stringList(SysConfKeys.host_whitelist.name()));
		if (list) {
			if (whitelist.isEmpty()) {
				ctx.info("No hosts are whitelisted");
			} else {
				ctx.info(String.join(",", whitelist));
			}
			System.exit(0);
		}

		if (remove && whitelist.remove(fqdn)) {
			update(service, whitelist);
		} else {
			if (whitelist.add(fqdn)) {
				update(service, whitelist);
			}
		}
	}

	private void update(ISystemConfiguration service, Set<String> whitelist) {
		Map<String, String> updated = Map.of(SysConfKeys.host_whitelist.name(),
				String.join(SystemConf.SYSTEMCONF_SEPARATOR, whitelist));
		service.updateMutableValues(updated);
	}

	@Override
	public int getExitCode() {
		return exitCode;
	}

	@Override
	public Runnable forContext(CliContext ctx) {
		this.ctx = ctx;
		return this;
	}

}
