/* BEGIN LICENSE
  * Copyright © Blue Mind SAS, 2012-2025
  *
  * This file is part of Blue Mind. Blue Mind is a messaging and collaborative
  * solution.
  *
  * This program is free software; you can redistribute it and/or modify
  * it under the terms of either the GNU Affero General Public License as
  * published by the Free Software Foundation (version 3 of the License)
  * or the CeCILL as published by CeCILL.info (version 2 of the License).
  *
  * There are special exceptions to the terms and conditions of the
  * licenses as they are applied to this program. See LICENSE.txt in
  * the directory of this program distribution.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
  *
  * See LICENSE.txt
  * END LICENSE
  */
package net.bluemind.lib.vertx.utils;

import java.time.Duration;
import java.util.function.Function;
import java.util.function.IntSupplier;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.RemovalCause;
import com.google.common.base.Supplier;
import com.google.common.base.Suppliers;

import io.vertx.core.Handler;
import io.vertx.core.Vertx;
import io.vertx.core.eventbus.Message;
import net.bluemind.lib.vertx.VertxPlatform;
import net.bluemind.lifecycle.helper.SoftReset;

public class ThrottleAccumulator<T> {
	private static final Logger logger = LoggerFactory.getLogger(ThrottleAccumulator.class);
	private static final Supplier<Boolean> DISABLED = Suppliers
			.memoize(() -> "true".equals(System.getProperty("throttle.disabled", "false")));
	private final IntSupplier throttleMs;
	private final Function<Message<T>, String> keyExtractor;
	private final Cache<String, ThrottleState<T>> throttleMap;
	private Vertx vertx;
	private Handler<Message<T>> wrappedHandler;

	private static class ThrottleState<T> {
		volatile long nextExecution;
		volatile Long timerId;
		volatile Message<T> pendingValue;

		public String toString() {
			return "timer=" + timerId + " lastExec: " + nextExecution + " value: " + pendingValue;
		}
	}

	public ThrottleAccumulator(Vertx vertx, Duration throttleMs, Function<Message<T>, String> keyExtractor,
			Handler<Message<T>> wrappedHandler) {
		this(vertx, () -> (int) throttleMs.toMillis(), keyExtractor, wrappedHandler);
	}

	public ThrottleAccumulator(Vertx vertx, IntSupplier throttleMs, Function<Message<T>, String> keyExtractor,
			Handler<Message<T>> wrappedHandler) {
		this.vertx = vertx;
		throttleMap = Caffeine.newBuilder()
				.expireAfterWrite(
						Duration.ofMillis(throttleMs.getAsInt()).plus(Duration.ofMillis(throttleMs.getAsInt())))
				.removalListener((String key, ThrottleState<T> ts, RemovalCause cause) -> {
					if (ts != null && ts.timerId != null) {
						vertx.cancelTimer(ts.timerId);
						ts.timerId = null;
					}
				}).evictionListener((String key, ThrottleState<T> ts, RemovalCause cause) -> {
					if (ts != null && ts.timerId != null) {
						vertx.cancelTimer(ts.timerId);
						ts.timerId = null;
					}
				}).build();
		this.throttleMs = throttleMs;
		this.keyExtractor = keyExtractor;
		this.wrappedHandler = wrappedHandler;
		SoftReset.register(() -> throttleMap.invalidateAll());
	}

	public void handle(Message<T> event) {
		if (DISABLED.get()) {
			if (!Boolean.getBoolean("junit")) {
				logger.info("disabled");
			}
			processEvent(event);
			return;
		}

		String key = keyExtractor.apply(event);
		if (key == null) {
			logger.error("ThrottleAccumulator called with null key. Event: {}", event, new Exception("null key"));
			return;
		}
		long throttleNanos = throttleMs.getAsInt() * 1000 * 1000L;
		long now = System.nanoTime();

		ThrottleState<T> state = throttleMap.get(key, k -> new ThrottleState<>());
		state.pendingValue = event;
		if (state.nextExecution < now) {
			state.nextExecution = now + throttleNanos;
			if (state.timerId != null) {
				vertx.cancelTimer(state.timerId);
				state.timerId = null;
			}
			processEvent(state.pendingValue);
			state.pendingValue = null;
		} else {
			if (state.timerId != null) {
				// Already scheduled
				return;
			}
			state.timerId = vertx.setTimer(Duration.ofNanos(throttleNanos).toMillis(), id -> {
				state.timerId = null;
				processEvent(state.pendingValue);
				state.nextExecution = System.nanoTime() + throttleNanos;
				state.pendingValue = null;
			});
		}
	}

	private void processEvent(Message<T> message) {
		if (message == null) {
			logger.info("Weird: message is empty", new Exception("empty message"));
			return;
		}
		if (VertxPlatform.isInEventLoopThread()) {
			vertx.executeBlocking(() -> {
				wrappedHandler.handle(message);
				return null;
			}, true);
		} else {
			wrappedHandler.handle(message);
		}
	}
}
