package net.bluemind.directory.hollow.datamodel.consumer.multicore.keydb;

import static net.bluemind.directory.hollow.datamodel.consumer.multicore.keydb.KeyDbMultiCoreDirectoryKeys.adressbookRecordHKey;
import static net.bluemind.directory.hollow.datamodel.consumer.multicore.keydb.KeyDbMultiCoreDirectoryKeys.directoryKey;
import static net.bluemind.directory.hollow.datamodel.consumer.multicore.keydb.KeyDbMultiCoreDirectoryKeys.versionKey;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.function.Consumer;
import java.util.function.Predicate;

import org.slf4j.LoggerFactory;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;

import io.lettuce.core.KeyValue;
import io.lettuce.core.LettuceFutures;
import io.lettuce.core.RedisFuture;
import io.lettuce.core.ScanIterator;
import io.lettuce.core.api.push.PushMessage;
import io.lettuce.core.codec.StringCodec;
import net.bluemind.directory.hollow.datamodel.AddressBookRecord;
import net.bluemind.directory.hollow.datamodel.AddressBookRecordIndexOnly;
import net.bluemind.directory.hollow.datamodel.OfflineAddressBook;
import net.bluemind.directory.hollow.datamodel.consumer.multicore.IMultiCoreDirectoryDeserializerStore;
import net.bluemind.keydb.common.ClientProvider;

public class KeyDbMultiCoreDirectoryDeserializerStore implements IMultiCoreDirectoryDeserializerStore {

	private org.slf4j.Logger logger = LoggerFactory.getLogger(KeyDbMultiCoreDirectoryDeserializerStore.class);

	private String domainUid;

	private RedisConnections connections;

	public KeyDbMultiCoreDirectoryDeserializerStore(String domainUid) {
		this.domainUid = domainUid;
		connections = RedisConnections.factory.get();
	}

	@Override
	public List<AddressBookRecord> getAllAdressBookRecords(long version, Predicate<AddressBookRecord> filter) {
		List<AddressBookRecord> result = new ArrayList<>();
		logger.debug("start scan");
		ScanIterator<KeyValue<String, AddressBookRecord>> it = ScanIterator.hscan(connections.getRecordCommand(),
				directoryKey(domainUid, version));
		while (it.hasNext()) {
			AddressBookRecord value = it.next().getValue();
			if (filter.test(value)) {
				result.add(value);
			}
		}
		logger.debug("iteration done {} items in result", result.size());
		return result;
	}

	@Override
	public void forEachIndex(long version, Consumer<AddressBookRecordIndexOnly> consumer) {
		ScanIterator<KeyValue<String, AddressBookRecordIndexOnly>> it = ScanIterator
				.hscan(connections.getIndexCommand(), directoryKey(domainUid, version));
		while (it.hasNext()) {
			consumer.accept(it.next().getValue());
		}
	}

	private final Cache<String, AddressBookRecord> fastLoad = Caffeine.newBuilder()
			.expireAfterWrite(30, TimeUnit.SECONDS).maximumSize(1024).build();

	@Override
	public AddressBookRecord getAdressBookRecord(long version, String uid) {
		String cacheKey = directoryKey(domainUid, version) + "#" + adressbookRecordHKey(uid);
		return fastLoad.get(cacheKey,
				k -> connections.getRecordCommand().hget(directoryKey(domainUid, version), adressbookRecordHKey(uid)));
	}

	public List<AddressBookRecord> batchGetAdressBookRecords(long version, Collection<String> uids) {
		List<RedisFuture<AddressBookRecord>> futures = new ArrayList<>();

		for (String deletedUid : Optional.ofNullable(uids).orElse(List.of())) {
			futures.add(connections.getAsyncRecordCommand().hget(directoryKey(domainUid, version),
					adressbookRecordHKey(deletedUid)));
		}
		LettuceFutures.awaitAll(5, TimeUnit.SECONDS, futures.toArray(new RedisFuture[futures.size()]));

		return futures.stream().map(f -> {
			try {
				return f.get();
			} catch (InterruptedException | ExecutionException e) {
				throw new RuntimeException("Error occurs while waiting for redis results", e);
			}
		}).toList();
	}

	@Override
	public OfflineAddressBook getOfflineAdressBookRecords(long version) {
		return connections.getOfflineRecordCommand()
				.get(KeyDbMultiCoreDirectoryKeys.offlineDirectoryKey(domainUid, version));
	}

	@Override
	public long getVersion() {
		String keyStr = connections.getVersionCommand().get(versionKey(domainUid));
		return keyStr != null ? Long.parseLong(keyStr) : 0;
	}

	@Override
	public void watchVersionUpdate(Runnable onUpdate) {
		connections.getVersionPubsubCommand().configSet("notify-keyspace-events",
				ClientProvider.NOTIFY_KEYSPACE_EVENTCONFIG);
		connections.getVersionPubsubCommand().subscribe(getSetVersionChannel());

		connections.getVersionPubsubConnection().addListener((PushMessage message) -> {
			String eventMessage;
			try {
				eventMessage = (String) message.getContent(StringCodec.ASCII::decodeKey).get(2);
			} catch (Exception e) {
				eventMessage = "NOT_SUPORTED_EVENT";
			}
			String eventType = message.getType();
			logger.debug("Receive {}/{}", eventType, eventMessage);
			if ("message".equals(eventType) && "set".equals(eventMessage)) {
				onUpdate.run();
			}
		});
	}

	@Override
	public void stopWatchVersionUpdate() {
		connections.getVersionPubsubCommand().unsubscribe(getSetVersionChannel());
	}

	@Override
	public boolean isWatchingVersionUpdate() {
		return !connections.getVersionPubsubCommand().pubsubChannels(getSetVersionChannel()).isEmpty();
	}

	private String getSetVersionChannel() {
		return new StringBuilder("__keyspace@0__:").append(KeyDbMultiCoreDirectoryKeys.versionKey(domainUid))
				.toString();
	}

}
